#include <fstream>
#include "MTGParser.H"

// The MTGParser constructor takes in an istream containing the MTG to
// be parsed.
MTGParser::MTGParser(std::istream& in)
  : mtgData() , topoMap() , labelMap() , aliasMap()
{
  // We first read in the MTG line-by-line, using istream::getline.
  static char line[4096];
  in.getline(line,4096);
  while(in)
  {
    // workstring is a C++ string representation of the current line.
    std::string workstring(line);

    // We will create a vector "row" of the strings in each column of
    // the current row.
    std::vector<std::string> row;

    // start_index is the character index within workstring of the
    // start of the current column.
    int start_index = 0;

    // We move through the workstring, extracting the text between
    // successive TABs (i.e., the columns) into the vector "row".
    while(start_index < workstring.size())
    {
      int tab_index = workstring.find('\t',start_index);
      if(tab_index == std::string::npos)
	tab_index = workstring.size();

      row.push_back(std::string(workstring,start_index,tab_index - start_index));
      start_index = tab_index + 1;
    }

    // We insert the row into the mtgData structure.
    mtgData.push_back(row);

    // and read the next line from the file.
    in.getline(line,1024);
  }


  // Now we parse the mtgData.
  // row is the index of the row we are currently processing.
  int row = 0;

  // First, we completely bypass the MTG header data, and go straight
  // for the TOPO data (or, rather, the first TOPO after the "MTG:"
  // declaration).
  for(bool flag = false ; row < mtgData.size() ; row++)
  {
    if(flag && mtgData[row].size() > 0 && mtgData[row][0] == "TOPO")
      break;
    else if(mtgData[row].size() > 0 && mtgData[row][0] == "MTG:")
      flag = true;
  }
  // Now row is the row index of the column headers.

  // Next, we read in the column labels. firstDataColumn will be the
  // index of the first column which contains parameter data (as
  // opposed to a topological structure string).
  int firstDataColumn = -1;
  for(int column = 1 ; column < mtgData[row].size() ; column++)
  {
    if(mtgData[row][column].size() > 0)
    {
      if(firstDataColumn < 0) firstDataColumn = column;

      // The column label is associated in the labelMap with the
      // column index.
      labelMap[mtgData[row][column]] = column;
    }
  }
  row++;

  // Now we identify which column contains the aliases. There doesn't
  // have to be one, but there'd better not be any aliases if there
  // isn't. aliasIndex will contain the index of this column.
  int aliasIndex = -1;
  const static std::string aliasName("Alias");
  if(labelMap.find(aliasName) != labelMap.end())
    aliasIndex = (*(labelMap.find(aliasName))).second;

  // Finally, we iterate through the TOPO section.

  // lastStrings holds the last string defined in a given column (for
  // use with the "^" character in the MTG, which appends its string
  // to the last defined string in the same column).
  std::vector<std::string> lastStrings(firstDataColumn);

  for( ; row < mtgData.size() ; row++)
  {
    // activeColumn is the last column which contains a plant
    // structure definition. 
    int activeColumn;
    for(activeColumn = Min(firstDataColumn - 1,int(mtgData[row].size()) - 1) ;
	activeColumn >= 0 ;
	activeColumn--)
      if(mtgData[row][activeColumn].size() > 0)
	break;

    // If there is no plant structure on this row, we go the next row.
    if(activeColumn < 0)
      continue;

    // First, if the new structure string starts with a '^', we
    // append it to the lastString from the same column.
    if(mtgData[row][activeColumn][0] == '^')
      lastStrings[activeColumn] +=
	mtgData[row][activeColumn].substr(1,mtgData[row][activeColumn].size());

    // If there is no '^', but there's a column to the left (which is
    // the parent of the current string), then we append the new
    // structure string to the lastString from the previous column.
    else if(activeColumn > 0)
      lastStrings[activeColumn] =
	lastStrings[activeColumn - 1] + mtgData[row][activeColumn];

    // Otherwise, there's no '^', and we're in the first column, so
    // the lastString is just the given structure string.
    else
      lastStrings[activeColumn] = mtgData[row][activeColumn];

    // The lastString may contain aliases, so we remove them.
    removeAliasesFrom(lastStrings[activeColumn]);

    // We associate this (canonical) structure string in the topoMap
    // with the index of the current row.
    topoMap[lastStrings[activeColumn]] = row;

    // Finally, if this row has an alias, then the alias is associated
    // in the aliasMap to the structure string.
    if(mtgData[row].size() > aliasIndex &&
       mtgData[row][aliasIndex].size() > 0)
      aliasMap[mtgData[row][aliasIndex]] = lastStrings[activeColumn];
  }
}

// The helper method removeAliasesFrom replaces aliases (set off by
// parentheses ()) in the given string with the corresponding
// canonical plant structure descriptors.
void MTGParser::removeAliasesFrom(std::string& str)
{
  // We iterate through the string, searching for parentheses (which
  // set off an alias), then replacing the alias with the
  // corresponding string.
  for(int openPar = str.find('('),closePar = str.find(')',openPar) ;
      openPar != std::string::npos && closePar != std::string::npos ;
      openPar = str.find('(',closePar+1) , closePar = str.find(')',openPar))
  {
    // The alias is the string between the parentheses.
    std::string alias = str.substr(openPar+1,closePar - openPar - 1);

    // We insert enough space to make up the difference between the
    // sizes of the alias and the corresponding canonical string.
    str.insert(openPar,aliasMap[alias].size() - (alias.size() + 2),' ');

    // Finally, we replace the characters with the canonial string.
    str.replace(openPar,aliasMap[alias].size(),aliasMap[alias]);
  }
}

// The method lookupValue takes two C++ strings: the first describes
// the topological structure of a plant component (as defined in the
// TOPO field, possibly including aliases); the second is the label of
// a data field (as defined in the FEATURES section). If the row
// defining the given plant component has an entry in the column of
// the given parameter, it is returned (as a string); otherwise, an
// empty string is returned.
std::string MTGParser::lookupValue(const std::string& topo ,
                                   const std::string& label)
{
  // First, we copy topo into topo1, then remove aliases (if any) from it.
  std::string topo1(topo);
  removeAliasesFrom(topo1);

  // topoIdx will contain the row index of the given structure
  // description; labelIdx will contain the column index of the given
  // parameter label.
  int topoIdx,labelIdx;

  // If the given structure string is not in the topoMap, or...
  if((topoMap.find(topo1) == topoMap.end())   ||
     // ...the given label is not in the labelMap, or...
     (labelMap.find(label) == labelMap.end()) ||
     // ...the given structure's row is shorter than the label's column...
     (mtgData[topoIdx = (*(topoMap.find(topo1))).second].size() <=
      (labelIdx = (*(labelMap.find(label))).second)))
    // ...then we return an empty string.
    return std::string();

  // Otherwise, we return the cell of the MTG at the corresponding row
  // and column indices.
  else
      return mtgData[topoIdx][labelIdx];
}


// The ostream output operator outputs the entire MTG. It will
// hopefully be identical to the original MTG.
std::ostream& operator<<(std::ostream& out,const MTGParser& mtg)
{
  // We simply iterate through the rows...
  for(int i = 0 ; i < mtg.mtgData.size() ; i++)
  {
    // ...and through the columns...
    for(int j = 0 ; j < mtg.mtgData[i].size() ; j++)
    {
      // outputing the cells, separated by TABs.
      if(j > 0)
	out << "\t";
      out << mtg.mtgData[i][j];
    }
    out << std::endl;
  }
  return out;
}


// Here, commented out, are the original functions I used
// to test the MTGParser class. They continue to the end
// of the file.
#if 0
int main(void)
{
  ifstream in("arabidopsis.mtg");
  MTGParser mtg(in);
  mtg.test();

  return 1;
}

#include <cctype>

void MTGParser::test(void)
{
  const int DN = 12;
  const int MN = (19 + 1);

  /*
  cout << "RM: { ";
  cout << atoi(lookupValue("/A0","RM").data()) << " ";
  string lookStr("/A0/M0");
  for(int i = 0 ; i < DN ; i++)
  {
    cout << atoi(lookupValue(lookStr+"/I0+A1","RM").data()) << " ";
    lookStr += "<M0";
  }
  cout << "};" << endl;
  */

  /*
  cout << "RD: { ";
  cout << atoi(lookupValue("/A0","RD").data()) << " ";
  string lookStr("/A0/M0");
  for(int i = 0 ; i < DN ; i++)
  {
    cout << atoi(lookupValue(lookStr+"/I0+A1","RD").data()) << " ";
    lookStr += "<M0";
  }
  cout << "};" << endl;
  */

  /*
  cout << "iangle: " << endl;
  string lookStr("/A0/M0");
  for(int i = 0 ; i < MN ; i++)
  {
    cout << atof(lookupValue(lookStr+"/I0","IA").data()) << endl;
    lookStr += "<M0";
  }
  lookStr.assign("/A0/M0");
  for(int n = 1 ; n < DN ; n++)
  {
    string subStr(lookStr + "/I0+A1/M1");
    for(int i = 0 ; i < MN ; i++)
    {
      cout << atof(lookupValue(subStr+"/I1","IA").data()) << endl;
      subStr += "<M1";
    }

    lookStr += "<M0";
  }
  cout << endl;
  */

  cout << "ilparameters: " << endl;
  string lookStr("/A0/M0");
  for(int i = 0 ; i < MN ; i++)
  {
    cout << atof(lookupValue(lookStr+"/I0","IL0").data()) << "\t";
    cout << atof(lookupValue(lookStr+"/I0","IL1").data()) << "\t";
    cout << atof(lookupValue(lookStr+"/I0","IL2").data()) << "\t";
    cout << atof(lookupValue(lookStr+"/I0","IL3").data()) << endl;
    lookStr += "<M0";
  }
  lookStr.assign("/A0/M0");
  for(int n = 1 ; n < DN ; n++)
  {
    string subStr(lookStr + "/I0+A1/M1");
    for(int i = 0 ; i < MN ; i++)
    {
      cout << atof(lookupValue(subStr+"/I1","IL0").data()) << "\t";
      cout << atof(lookupValue(subStr+"/I1","IL1").data()) << "\t";
      cout << atof(lookupValue(subStr+"/I1","IL2").data()) << "\t";
      cout << atof(lookupValue(subStr+"/I1","IL3").data()) << endl;
      subStr += "<M1";
    }

    lookStr += "<M0";
  }
  cout << endl;
}
#endif // 0

