#ifndef __MTG_PARSER
#define __MTG_PARSER

#include <iostream>
#include <string>
#include <vector>
#include <map>

/****
 * The MTGParser class reads in and parses an MTG from a supplied
 * istream, and supplies a method for looking up data values from the
 * TOPO section of the MTG.
 ****/
class MTGParser
{
public:
  // The constructor takes in an istream containing the MTG
  // to be parsed.
  MTGParser(std::istream& in);

  // The method lookupValue takes two C++ strings: the first describes
  // the topological structure of a plant component (as defined in the
  // TOPO field, possibly including aliases); the second is the label
  // of a data field (as defined in the FEATURES section). If the row
  // defining the given plant component has an entry in the column of
  // the given parameter, it is returned (as a string); otherwise, an
  // empty string is returned.
  std::string lookupValue(const std::string& topo,
			  const std::string& label);

  // The ostream output operator outputs the entire MTG. It will
  // hopefully be identical to the original MTG.
  friend std::ostream& operator<<(std::ostream& out,const MTGParser& mtg);


  // The test method tests some features of the MTG (how to extract
  // axial and metamer parameters, on both primary and secondary axes).
  void test(void);

private:
  // The helper method removeAliasesFrom replaces aliases (set off by
  // parentheses ()) in the given string with the corresponding
  // canonical plant structure descriptors.
  void removeAliasesFrom(std::string& str);

  // cmpstr is a function object which compares two C++ strings; it
  // is used in the definition of IndexMap and StringMap below.
  struct cmpstr
  {
    bool operator()(const std::string& s1,const std::string& s2) const
    {
      return (s1 < s2);
    }
  };

  // The MTG is split up into rows and columns (with columns separated
  // by TABs). The vector-of-vectors mtgData holds each "cell" of the
  // MTG; each element of it is a vector holding the cells in a single
  // row.
  std::vector< std::vector<std::string> > mtgData;

  // The aliases are kept in a map from a string (the alias) to
  // a string (the canonical topology string).
  typedef std::map< std::string , std::string , cmpstr > StringMap;
  StringMap aliasMap;

  // We also maintain a pair of maps which relate strings to indices;
  // topoMap relates a canonical topology string to a row index in
  // the mtgData data structure; labelMap relates a parameter label
  // to a column index.
  typedef std::map< std::string , int , cmpstr > IndexMap;
  IndexMap topoMap,labelMap;
};

#endif // __MTG_PARSER
