#ifndef __VECTOR_H
#define __VECTOR_H

#include <iostream>
#include <cmath>

template <int dim,class T = double>
class Vector
{
protected:
 T elems[dim];

public:
 Vector(void)
 {
   for(int i = 0 ; i < dim ; i++)
     elems[i] = T();
 }
 
 Vector(const Vector& vec)
 {
   for(int i = 0 ; i < dim ; i++)
     elems[i] = vec.elems[i];
 }
 
 Vector(const T* el)
 {
   for(int i = 0 ; i < dim ; i++)
     elems[i] = el[i];
 }
 

 Vector operator-(void) const
 {
   Vector ans;
   for(int i = 0 ; i < dim ; i++)
     ans[i] = -elems[i];
 
   return ans;
 }
 
 
 Vector operator+(const Vector& vec) const
 {
   Vector ans;
   for(int i = 0 ; i < dim ; i++)
     ans[i] = elems[i] + vec.elems[i];
 
   return ans;
 }
 
 Vector operator-(const Vector& vec) const
 {
   Vector ans;
   for(int i = 0 ; i < dim ; i++)
     ans[i] = elems[i] - vec.elems[i];
 
   return ans;
 }
 
 
 Vector operator*(const T& scalar) const
 {
   Vector ans;
   for(int i = 0 ; i < dim ; i++)
     ans[i] = elems[i] * scalar;
 
   return ans;
 }
 
 Vector operator/(const T& scalar) const
 {
   Vector ans;
   for(int i = 0 ; i < dim ; i++)
     ans[i] = elems[i] / scalar;
 
   return ans;
 }
 
 
 friend Vector operator*(const T& scalar,const Vector& vec)
 {
   Vector ans;
   for(int i = 0 ; i < dim ; i++)
     ans[i] = scalar * vec.elems[i];
 
   return ans;
 }
 
 
 T operator*(const Vector& vec) const
 {
   T ans = 0;
   for(int i = 0 ; i < dim ; i++)
     ans += elems[i] * vec.elems[i];
 
   return ans;
 }
 
 
 Vector& operator=(const Vector& vec)
 {
   for(int i = 0 ; i < dim ; i++)
     elems[i] = vec.elems[i];
 
   return (*this);
 }
 
 Vector& operator+=(const Vector& vec)
 {
   return ((*this) = (*this) + vec);
 }
 
 Vector& operator-=(const Vector& vec)
 {
   return ((*this) = (*this) - vec);
 }
 
 Vector& operator*=(const T& scalar)
 {
   return ((*this) = (*this) * scalar);
 }
 
 Vector& operator/=(const T& scalar)
 {
   return ((*this) = (*this) / scalar);
 }
 
 
 bool operator==(const Vector& vec) const
 {
   for(int i = 0 ; i < dim ; i++)
     if(elems[i] != vec.elems[i])
       return false;
 
   return true;
 }
 
 bool operator!=(const Vector& vec) const
 {
   return (!((*this) == vec));
 }
 
 
 friend std::ostream& operator<<(std::ostream& out,const Vector& vec)
 {
   out << "(";
   for(int i = 0 ; i < dim ; i++)
   {
     out << vec.elems[i];
     if(i != (dim - 1))
       out << ",";
   }
   out << ")";
 
   return out;
 }
 
 friend std::istream& operator>>(std::istream& in,Vector& vec)
 {
   char delim;
   in >> delim;
   if(delim != '(')
   {
     in.putback(delim);
     in.clear(std::ios::badbit|in.rdstate());
     return in;
   }
 
   for(int i = 0 ; i < dim && delim != ')' && in ; i++)
     in >> vec[i] >> delim;
 
   return in ;
 }
 
 T& operator[](int idx)
 {
   return elems[idx];
 }
 
 T operator[](int idx) const
 {
   return elems[idx];
 }
 
 
 
 T size(void) const
 {
   return sqrt(size2());
 }
 
 T size2(void) const
 {
   T ans = 0;
   for(int i = 0 ; i < dim ; i++)
     ans += elems[i] * elems[i];
 
   return ans;
 }
 
 Vector& normalize(void)
 {
   static T EPSILON = (T) 1.0e-6;
 
   T sz = size();
   if(sz < EPSILON)
     return ((*this) = zero());
   else
     return ((*this) /= sz);
 }
 
 Vector normalized(void) const
 {
   Vector ans(*this);
   return ans.normalize();
 }
 
 bool isZero(void)
 {
   for(int i = 0 ; i < dim ; i++)
     if(elems[i] != 0)
       return false;
   return true;
 }
 
 
 static Vector axis(int idx)
 {
   Vector ans;
   if(idx >= 0 && idx < dim)
     ans.elems[idx] = 1;
 
   return ans;
 }
 
 static Vector zero(void)
 {
   Vector ans;
   return ans;
 }
 
 
};


class Vector2 : public Vector<2,double>
{
public:
  Vector2(void) : Vector<2,double>() {}
  
  Vector2(double x,double y)
    : Vector<2,double>()
  {
    elems[0] = x;
    elems[1] = y;
  }
  
  Vector2(const Vector<2,double>& vec)
    : Vector<2,double>(vec)
  {}
  
  

  double operator&(const Vector2& vec) const
  {
    return ((elems[0] * vec.elems[1]) -
            (elems[1] * vec.elems[0]));
  }
  
  

  Vector2 perp(void)
  {
    return Vector2(elems[1],-elems[0]);
  }
  
  double& x(void) { return elems[0]; }
  double& y(void) { return elems[1]; }
  double x(void) const { return elems[0]; }
  double y(void) const { return elems[1]; }
  
  
};

extern const Vector2 XAXIS_2;
extern const Vector2 YAXIS_2;
extern const Vector2 ZERO_2;


class Vector3 : public Vector<3,double>
{
public:
  Vector3(void) : Vector<3,double>() {}
  
  Vector3(const Vector<3,double>& _vec)
    : Vector<3,double>(_vec)
  {}
  
  Vector3(double x,double y,double z)
    : Vector<3,double>()
  {
    elems[0] = x;
    elems[1] = y;
    elems[2] = z;
  }
  
  Vector3(const Vector2& vec)
    : Vector<3,double>()
  {
    elems[0] = vec[0];
    elems[1] = vec[1];
    elems[2] = 0;
  }
  
  Vector3(const Vector2& vec,double z)
    : Vector<3,double>()
  {
    elems[0] = vec[0];
    elems[1] = vec[1];
    elems[2] = z;
  }
  
  

  Vector2 projectXY(void)
  {
    return Vector2(elems[0],elems[1]);
  }
  
  double& x(void) { return elems[0]; }
  double& y(void) { return elems[1]; }
  double& z(void) { return elems[2]; }
  double x(void) const { return elems[0]; }
  double y(void) const { return elems[1]; }
  double z(void) const { return elems[2]; }
  
  
};

template<class T>
Vector<3,T> operator&(const Vector<3,T>& v1,const Vector<3,T>& v2)
{
  Vector<3,T> ans;
  ans[0] = v1[1]*v2[2] - v1[2]*v2[1];
  ans[1] = v1[2]*v2[0] - v1[0]*v2[2];
  ans[2] = v1[0]*v2[1] - v1[1]*v2[0];

  return ans;
}


#endif // __VECTOR_H
