function ModelViewer(varargin)
% Dr. A. I. Hanna (2005) CMP, UEA, 2005.
error(nargchk(0,inf,nargin));
ad.options.pdm = '';
ad.options.sfam = '';
ad.options.PMdirectory = '';
ad.options.ImageDirectory = '';
ad.options.savePDM = 0;
ad.pdm = [];
ad.edges = [];
ad.options.ref_image = [];
ad.pmpts = [];
ad.indices =[];
if isProject3D(pwd)
    ad.dim = 3;
else
    ad.dim = 2;
end
if mod(length(varargin),2) ~= 0
    % input args have not com in pairs, woe is me
    error('Arguments to ModelViewer must come param/value in pairs.')
end
for i=1:2:length(varargin)
    switch lower(varargin{i})
        case 'pdmfile'
            ad.options.pdm = varargin{i+1};
        case 'sfamfile'
            ad.options.sfam = varargin{i+1};
        case 'pmdirectory'
            ad.options.PMdirectory = varargin{i+1};
        case 'imdirectory'
            ad.options.ImageDirectory = varargin{i+1};
        case 'edges'
            ad.edges = varargin{i+1};
        case 'details'
            ad.details = varargin{i+1};
        case 'indices'
            ad.indices = varargin{i+1};
        otherwise
            error(['Unknown parameter name passed to ModelViewer.  Name was ' varargin{i}])
    end
end
if isempty(ad.options.pdm)
    ad = manualInitialization(ad); 
end
% Open, move, and get the handles from the figure file.
fig = openfig(mfilename, 'reuse');
customtoolbar(fig);
% Move the gui and then show it, rather than have it jump all over the
% place.
movegui(fig, 'center');
set(fig, 'visible', 'on');
handles = guihandles(fig);
handles = SetGUILevel(handles, getGUILevelInfo);

[modelDirec, pdmfile, ext, vers] = fileparts(ad.options.pdm);
if findstr(modelDirec, pwd)==1
    modelDirec = modelDirec(length(pwd)+1:end);
end

set(handles.modelNameTxt, 'String', modelDirec);
set(fig, 'Color', get(handles.modelPanel, 'BackgroundColor'))
handles.app_modes_val = [];
handles.shape_modes_val = [];

% Set all the callback functions
set(handles.shape_pc_slider, 'callback', {@doUpdateShapeMode});
set(handles.app_pc_slider, 'callback', {@doUpdateAppMode});

% The shape controls
set(handles.walk_curr_shape_pc_btn, 'callback', {@doWalkCurrentShapePC});
set(handles.walk_all_shape_pc_btn, 'callback', {@doWalkAllShapePC});
set(handles.reset_shape_btn, 'callback', {@doResetShape});
set(handles.reset_all_shape_btn, 'callback', {@doResetAllShape});
set(handles.shapeslider, 'callback', {@doSlideModes, handles.shapelevels});
set(handles.shapelevels, 'ButtonDownFcn', {@doSelectMode, handles.shapeslider, 'shape'});

% The appearance controls
set(handles.walk_curr_app_pc_btn, 'callback', {@doWalkCurrentAppPC});
set(handles.walk_all_app_pc_btn, 'callback', {@doWalkAllAppPC});
set(handles.reset_app_btn, 'callback', {@doResetApp});
set(handles.reset_all_app_btn, 'callback', {@doResetAllApp});
set(handles.appslider, 'callback', {@doSlideModes, handles.applevels});
set(handles.applevels, 'ButtonDownFcn', {@doSelectMode, handles.appslider, 'app'});

% Handles to the rest of the GUI
set([handles.orient_ij_rad, handles.orient_xy_rad], 'callback', {@doUpdateDisplay});
set([handles.edges_chk, handles.show_landmarks_chk, handles.show_app_model_chk, handles.tps_chk], 'callback', {@doUpdateDisplay});
set(handles.save_image_btn, 'callback', {@doSaveImage});
set(handles.load_pm_file_btn, 'callback', {@doLoadPointModelFile});
set(handles.calc_weights_btn, 'callback', {@doCalcWeights});
set(handles.import_mean_shape_btn, 'callback', {@doImportMeanShape});
set(handles.add_rand_axis_btn, 'callback', {@doAddRandomShapeAxis});
set(handles.reload_model_btn, 'callback', {@doReloadModel});
set([handles.shape_range, handles.app_range], 'callback', {@doUpdateWalkerLimits});
set([handles.cart_axes_chk], 'callback', {@doChangeAxes});
set(handles.offsetchk, 'callback', {@doToggleOffset});


% Handles to the menus
set(handles.importshapepcmenu, 'callback', {@doImportShapePCMenu});
set(handles.exportshapepcmenu, 'callback', {@doExportShapePCMenu});
set(handles.flipshapepcmenuitem, 'callback', {@doFlipShapePC});
set(handles.importrefimage, 'callback', {@doImportRefImage});
set(handles.exportpmmenu, 'callback', {@doExportPointModel});


% Initialize the application data structure
ad.figMain = fig;
ad.handles = handles;
ad.app_levels_plot_handle = [];
ad.currentShapeMode = [];
ad.currentAppMode = [];

ad.data.edgehandle = [];
ad.data.shape_pts_handle = [];
ad.data.appearancehandle = [];

ad.currentImage = [];
[ad.segmentPoint, ad.offsets] = isAugmentedProject;
ad.dy= 15;
ad.shapewalkerlim = 3;
ad.appwalkerlim = 3;

ad = loadModels(ad);
setappdata(0,'ModelViewerData',ad);

initgui;
try
    uiwait(fig);
catch
    if ishandle(fig)
        delete(fig)
    end
end
if isappdata(0,'ModelViewerData')
    ad = getappdata(0,'ModelViewerData');
    if ad.options.savePDM
        ad = savePDM(ad);
    end
    rmappdata(0,'ModelViewerData')
else
    % figure was deleted
    selection = [];
    value = 0;
end
return;

%%%%%
%
%
%%%%%
function doUpdateDisplay(axis_radio, evd) %#ok<INUSD,INUSD>
ad = getappdata(0,'ModelViewerData');
set(ad.figMain, 'Pointer', 'watch');
drawnow;
plotModel;
title(ad.handles.mainAxes, '', 'Interpreter', 'none');

set(ad.figMain, 'Pointer', 'arrow');
return;
%%%%%
%
%
%%%%%
function ad = loadModels(ad)
opt = ad.options;
% Load the pdm if it exists
pdmfile = opt.pdm;
ad.pdmfilename = pdmfile;
if exist(ad.pdmfilename, 'file')
    ad.pdm = load(ad.pdmfilename);
    ad.pdm = ad.pdm.pdm;
    if isfield(ad.pdm, 'edges') && isempty(ad.edges)
        ad.edges = ad.pdm.edges;
    end
    if get(ad.handles.cart_axes_chk, 'Value')
        ad.pdm.P = eye(size(ad.pdm.P,1));
        ad.pdm.b = ones(size(ad.pdm.P,1), 1);
    end
    ad.handles.shape_modes_val = zeros(length(ad.pdm.b),1);
    ad.currentShapeModel = getCurrShapeModel(ad.pdm,  ad.handles.shape_modes_val, ad.dim);
end

% Load the sfam if it exists
sfamfile = opt.sfam;
if ~isempty(sfamfile)
    ad.sfamfilename = sfamfile;
    set(ad.handles.no_app_txt, 'Visible', 'off');
    if exist(ad.sfamfilename, 'file')
        ad.sfam = load(ad.sfamfilename);
        ad.sfam = ad.sfam.sfam;
        ad.handles.app_modes_val = zeros(length(ad.sfam.b),1);
    end
else
    ad.handles = toggleAppearanceGUI(ad.handles, 'off');
end
ad = setupSliderLimits(ad);
return;
%%%%%
%
%
%%%%%
function ad = setupSliderLimits(ad)
app_y_lim = length(ad.handles.app_modes_val) - ad.dy;
shape_y_lim = length(ad.handles.shape_modes_val) - ad.dy;
if -app_y_lim>=-1
    set(ad.handles.appslider, 'Enable', 'off');
else
    set(ad.handles.appslider, 'Min', -app_y_lim-1);
end
if -shape_y_lim>=-1
    set(ad.handles.shapeslider, 'Enable', 'off');
else
    set(ad.handles.shapeslider, 'Min', -shape_y_lim-1);
end
ad.currentShapeMode = 1;
ad.currentAppMode = 1;
return;
%%%%%
%
%
%%%%%
function togglePlotView
ad =getappdata(0, 'ModelViewerData');
if get(ad.handles.orient_xy_rad, 'Value');
    axis(ad.handles.mainAxes, 'xy');
else
    axis(ad.handles.mainAxes, 'ij');
end
setappdata(0, 'ModelViewerData', ad);
%%%%%
%
%
%%%%%
function ad = updateAxisLimits(ad, pts)
xrange = [min(pts(:,1)), max(pts(:,1))];
yrange = [min(pts(:,2)), max(pts(:,2))];
if xrange(1)<ad.xrange(1); ad.xrange(1) = xrange(1); end;
if xrange(2)>ad.xrange(2); ad.xrange(2) = xrange(2); end;
if yrange(1)<ad.yrange(1); ad.yrange(1) = yrange(1); end;
if yrange(2)>ad.yrange(2); ad.yrange(2) = yrange(2); end;
% Begin change by js, 070705
% Begin original code
 
% No replacement in this case 
 
% End original code
% Begin replacement code
 
if ad.dim == 3
    zrange = [min(pts(:,3)), max(pts(:,3))];
    if zrange(1)<ad.zrange(1); ad.zrange(1) = zrange(1); end;
    if zrange(2)>ad.zrange(2); ad.zrange(2) = zrange(2); end;
end
 
% End replacement code
% End change by js, 070705

return;
%%%%%
%
%
%%%%%
function initgui
ad =getappdata(0, 'ModelViewerData');
if isempty(ad.pdm)
    uiwait(msgbox('Could not find a shape model. Aborting.','No PDM.','modal'));
    close(ad.figMain);
    return;
end
Xm = ad.pdm.Xm;
mean_pts = reshape(Xm, ad.dim , length(Xm)/ad.dim )';
ad.xrange = [min(mean_pts(:,1)), max(mean_pts(:,1))];
ad.yrange = [min(mean_pts(:,2)), max(mean_pts(:,2))];
% Begin change by js, 070705
% Begin original code
 
% No replacement in this case 
 
% End original code
% Begin replacement code
 
if ad.dim == 3
    ad.zrange = [min(mean_pts(:,3)), max(mean_pts(:,3))];
end
 
% End replacement code
% End change by js, 070705
displayModes(ad.handles.shapelevels, ad.handles.shape_modes_val, ad.handles.shapeslider, 'shape');
ad = updateShapeWalkerLimits(ad);
if ~isempty(ad.options.sfam)
    displayModes(ad.handles.applevels, ad.handles.app_modes_val, ad.handles.appslider, 'app');
    ad = updateAppWalkerLimits(ad);
end
setappdata(0,'ModelViewerData',ad);
plotModel;
ad =getappdata(0, 'ModelViewerData');
axis(ad.handles.mainAxes, 'tight');
ad.xrange = get(ad.handles.mainAxes, 'Xlim');
ad.yrange = get(ad.handles.mainAxes, 'Ylim');
% Begin change by js, 070705
% Begin original code
 
% No replacement in this case 
 
% End original code
% Begin replacement code
 
if ad.dim == 3
    ad.zrange = get(ad.handles.mainAxes, 'Zlim')
end
 
% End replacement code
% End change by js, 070705


setappdata(0,'ModelViewerData',ad);
%%%%%
%
%
%%%%%
function doFlipShapePC(flippcmenu, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
if isempty(ad.pdm)
    uiwait(msgbox('The is no shape model currently loaded, aborting.','Error','modal'));
    return;
end
N = size(ad.pdm.P, 2);
str = cell(N,1);
for i=1:N; str{i} = num2str(i); end;
[s,v] = listdlg('Name', 'Direction Flip', 'PromptString','Select PC''s to flip', 'ListSize', [400, 200], 'SelectionMode','multiple', 'ListString',str);
if isempty(s)
    return;
end
ad.pdm.P(:,s) = -1*ad.pdm.P(:,s);
ad.pdm.pca.P(:,s) = -1*ad.pdm.pca.P(:,s);
ad.options.savePDM = 1;
setappdata(0, 'ModelViewerData', ad);
%%%%%
%
%
%%%%%
function doExportPointModel(exportpmmenu, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
[filename, pathname] = uiputfile('*.mat', 'Save as...');
if isequal(filename,0) || isequal(pathname,0)
    disp('User pressed cancel')
    return;
end
pts = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val, ad.dim)';
pts = pts(:);
[junk, filename] = fileparts(filename);
filename = [pathname, filesep, filename, '_pm.mat'];
save(filename, 'pts');


%%%%%
%
%
%%%%%
function doExportShapePCMenu(exportshapemenu, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
[filename, pathname] = uiputfile('*.xls', 'Save as...');
if isequal(filename,0) || isequal(pathname,0)
    disp('User pressed cancel')
    return;
end
b = ad.handles.shape_modes_val.*sqrt(ad.pdm.b);
I = cell(length(b),1);
for i=1:length(b)
    I{i,1} = b(i);
end
[success, message] = xlswrite([pathname, filesep, filename], I);
%%%%%
%
%
%%%%%
function doImportShapePCMenu(importshapemenu, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
[filename, pathname] = uigetfile('*.xls', 'Pick an Excel file');
if isequal(filename,0) || isequal(pathname,0)
    disp('User pressed cancel')
    return;
end
pcvals = xlsread([pathname, filesep, filename]);
pcvals = pcvals(:,1);
N = length(ad.handles.shape_modes_val);
if length(pcvals)>N
    pcvals = pcvals(1:N);
end
b = zeros(N,1);
b(1:length(pcvals)) = pcvals;
ad.handles.shape_modes_val = b./sqrt(ad.pdm.b);
%ad.handles.shape_modes_val = b;
doPlotModes(ad.handles.shapelevels, ad.handles.shape_modes_val, 'shape');
setappdata(0, 'ModelViewerData', ad);
plotModel;
%%%%%
%
%
%%%%%
function doChangeAxes(cartchk, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
ad = loadModels(ad);
setappdata(0, 'ModelViewerData', ad);
initgui;
%%%%
%
%
%%%%
function doToggleOffset(offsetchk, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
if get(ad.handles.offsetchk, 'Value')
    [ad.segmentPoint, ad.offsets] = isAugmentedProject;
else
    ad.offsets = zeros(length(ad.segmentPoint)+1, 3);

end
setappdata(0, 'ModelViewerData', ad);
updatePlot;
ad =getappdata(0, 'ModelViewerData');
axis(ad.handles.mainAxes, 'tight');
ad.xrange = get(ad.handles.mainAxes, 'Xlim');
ad.yrange = get(ad.handles.mainAxes, 'Ylim');
% Begin change by js, 070705
% Begin original code
 
% No replacement in this case 
 
% End original code
% Begin replacement code
 
if ad.dim == 3
    ad.zrange = get(ad.handles.mainAxes, 'Zlim')
end
 
% End replacement code
% End change by js, 070705

setappdata(0, 'ModelViewerData', ad);

%%%%%
%
%
%%%%%
function doUpdateWalkerLimits(txt, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
ad = updateShapeWalkerLimits(ad);
ad = updateAppWalkerLimits(ad);
setappdata(0, 'ModelViewerData', ad);
plotModel;

%%%%%
%
%
%%%%%
function ad = updateShapeWalkerLimits(ad)
lim = str2num(get(ad.handles.shape_range, 'String')); %#ok<ST2NM>
ad.shapewalkerlim = lim;
sh = ad.handles.shape_pc_slider;
val = get(sh, 'Value');
set(sh, 'Min', -abs(lim), 'Max', abs(lim));
if abs(val)>=abs(lim)
    set(sh, 'Value', sign(val)*lim);
end
%%%%%
%
%
%%%%%
function ad = updateAppWalkerLimits(ad)
lim = str2num(get(ad.handles.app_range, 'String')); %#ok<ST2NM>
ad.appwalkerlim = lim;
sh = ad.handles.app_pc_slider;
val = get(sh, 'Value');
set(sh, 'Min', -abs(lim), 'Max', abs(lim));
if abs(val)>=abs(lim)
    set(sh, 'Value', sign(val)*lim);
end
%%%%%
%
%
%%%%%
function doReloadModel(reloadModelbtn, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
ad = loadModels(ad);
ad.options.ref_image = [];
ad.FLIPIND = [];
setappdata(0,'ModelViewerData',ad);
initgui;
%%%%%
%
%
%%%%%
function doAddRandomShapeAxis(randaxisbtn, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
P = ad.pdm.pca.P;
b = ad.pdm.b;
p = P*randn(size(P,2),1);
p = p/norm(p);
ad.pdm.P = [ad.pdm.P, p];
ad.pdm.b = [b; b(1)];
ad.handles.shape_modes_val = [ad.handles.shape_modes_val; 0];
setappdata(0, 'ModelViewerData', ad);
ad = setupSliderLimits(ad);
axis_h = ad.handles.shapelevels;
doPlotModes(axis_h, ad.handles.shape_modes_val, 'shape');
%%%%%
%
%
%%%%%
function doImportMeanShape(importmeanshpbtn, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
pmdir = [ad.options.PMdirectory];
[filename, pathname] = uigetfile('*_pm.mat', 'Pick an PM-file', [pmdir, filesep]);
if isequal(filename,0) || isequal(pathname,0)
    disp('User pressed cancel');
    return;
else
    meanshape = load([pathname, filesep, filename]);
    meanshape = meanshape.pts;
    disp(['User selected ', fullfile(pathname, filename)])
end
ad.pdm.Xm = meanshape;
setappdata(0, 'ModelViewerData', ad);
plotModel;
%%%%%
%
%
%%%%%
function doCalcWeights(calcweightsbtn, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
%%%%%%%%
% Take care of the shape first
%%%%%%%%
if isempty(ad.pmpts)
    uiwait(msgbox('Please load a point model file first','No pm file loaded','modal'));
    return;
end
details = ad.details;
original_pts = ad.pmpts;
%[ad.pmpts] = pcalib_GPA('data', ad.pmpts, 'opts', [details.scaling,details.rotation,details.translation], 'dimension', ad.dim, 'template', ad.pdm.Xm);
[ad.pmpts] = pcalib_GPA('data', ad.pmpts, 'opts', details, 'dimension', ad.dim, 'template', ad.pdm.Xm);
%ad.pmpts=pmalign(ad.pmpts,[details.scaling,details.rotation,details.translation],ad.pdm.Xm);
b = ad.pdm.P'*(ad.pmpts - ad.pdm.Xm);
ad.handles.shape_modes_val = ((b./ (sqrt(ad.pdm.b))));
doPlotModes(ad.handles.shapelevels, (ad.handles.shape_modes_val), 'shape');
%%%%%%%%
% Now its time for the appearance
%%%%%%%%
original_pts = reshape(original_pts, ad.dim , length(original_pts)/ad.dim );
if isfield(ad, 'sfam')
    if ~isempty(ad.sfam)
        meansize_im = triwarp(ad.pmimage, original_pts', ad.sfam.ompts); %warped image
        sz = ad.sfam.siz;
        Am = ad.sfam.Am;
        P = ad.sfam.P;
        b = ad.sfam.b;
        ompts = ad.sfam.ompts;
        model_b = P'*(meansize_im(:) - Am);
        ad.handles.app_modes_val = model_b./sqrt(ad.sfam.b);
        doPlotModes(ad.handles.applevels, ad.handles.app_modes_val, 'app');
        setappdata(0, 'ModelViewerData', ad);
    end
end
setappdata(0, 'ModelViewerData', ad);
plotModel;
title(ad.handles.mainAxes, ['Model - ', ad.modelledimagename], 'Interpreter', 'none');

%%%%%
%
%
%%%%%
function doLoadPointModelFile(loadpmfilebtn, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
pmdirectory = ad.options.PMdirectory;

%[filename, pathname] = uigetfile({'*.mat'}, 'Pick an PM file', [pmdirectory, filesep]);
[filename, pathname] = uigetfile({'*.jpg;*.JPG';'*.jpeg;*.JPEG'}, 'Pick an JPEG-file', ['Cropped', filesep]);

if isequal(filename,0) || isequal(pathname,0)
    disp('User pressed cancel')
    return;
else
    disp(['User selected ', fullfile(pathname, filename)])
end
pmdirectory = ad.options.PMdirectory;
pmfile = getPMFile(filename, pmdirectory);
%pmfile = filename;
if isempty(pmfile)
    uiwait(msgbox('There is no PM file for your chosen image','No pm file found','modal'));
    return;
end

pts = load([pmdirectory, filesep, pmfile]);
ad.pmpts = pts.pts;
if ~isempty(ad.indices)
    ad.pmpts = reshape(ad.pmpts, ad.dim, length(ad.pmpts)/ad.dim);
    ad.pmpts = ad.pmpts(:, ad.indices);
    ad.pmpts = ad.pmpts(:);
end
[pmdir, pmfile, ext, vers] = fileparts(pmfile);
pmfile = pmfile(1:end-3);
imname = ['Cropped', filesep, pmfile, '.jpg'];
if exist(imname, 'file')
    ad.pmimage = double(imread(imname));
    fig = figure;
    imshow(uint8(ad.pmimage));
    title(gca, ['Original - ', pmfile], 'Interpreter', 'none');
    set(fig, 'Position', [0 0 500 500]);

else
    ad.pmimage = [];
end
ad.modelledimagename = pmfile;
setappdata(0, 'ModelViewerData', ad);
%%%%%
%
%
%%%%%
function pmfilename = getPMFile(imname, D)
files = dir([D, filesep, '*_pm.mat']);
files = {files.name};
pmfilename = [];
[impath, imname, ext, vers] = fileparts(imname);
for i=1:length(files)
    pmfile = files{i};
    [pmpath, pmfile, ext, vers] = fileparts(pmfile);
    pmfile = pmfile(1:end-3);
    if strcmp(imname, pmfile)
        pmfilename = files{i};
        return;
    end
end
return;
%%%%%
%
%
%%%%%
function doSaveImage(savebtn, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');

[filename, pathname] = uiputfile({'*.tiff', 'TIFF Image (*.tiff)'}, 'Save as...');
if isequal(filename,0) || isequal(pathname,0)
    disp('User pressed cancel')
else
    disp(['User selected ', fullfile(pathname, filename)])
    %set(ad.figMain, 'InvertHardcopy', 'off');
    %print(ad.figMain,'-dtiffnocompression',[pathname, filesep, filename]);
    I = frame2im(getframe(ad.handles.mainAxes));
    imwrite(I, [pathname, filesep, filename], 'TIFF');
end
setappdata(0, 'ModelViewerData', ad);
%%%%%
%
%
%%%%%
function displayModes(axis_h, modes_val, slider_h, type)
doPlotModes(axis_h, modes_val, type);
highlightMode(1, axis_h);
doSlideModes(slider_h, [], axis_h);
%%%%%
%
%
%%%%%
function doResetShape(reset, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
val = ad.currentShapeMode;
ad.handles.shape_modes_val(val) = 0;
set(ad.handles.shape_pc_slider, 'Value', 0);
setappdata(0, 'ModelViewerData', ad);
plotModel;
doPlotModes(ad.handles.shapelevels, ad.handles.shape_modes_val, 'shape');
if val<=length(ad.handles.shape_modes_val)
    highlightMode(val, ad.handles.shapelevels);
end
%%%%%
%
%
%%%%%
function doResetApp(reset, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
val = ad.currentAppMode;
ad.handles.app_modes_val(val) = 0;
set(ad.handles.app_pc_slider, 'Value', 0);
setappdata(0, 'ModelViewerData', ad);
plotModel;
doPlotModes(ad.handles.applevels, ad.handles.app_modes_val, 'app');
if val<=length(ad.handles.app_modes_val)
    highlightMode(val, ad.handles.applevels);
end
%%%%%
%
%
%%%%%
function doResetAllShape(reset, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
val = ad.currentShapeMode;
ad.handles.shape_modes_val = zeros(size(ad.handles.shape_modes_val));
set(ad.handles.shape_pc_slider, 'Value', 0);
setappdata(0, 'ModelViewerData', ad);
plotModel;
doPlotModes(ad.handles.shapelevels, ad.handles.shape_modes_val, 'shape');
if val<=length(ad.handles.shape_modes_val)
    highlightMode(val, ad.handles.shapelevels);
end
%%%%%
%
%
%%%%%
function doResetAllApp(reset, evd) %#ok<INUSD,INUSD>
ad =getappdata(0, 'ModelViewerData');
val = ad.currentAppMode;
ad.handles.app_modes_val = zeros(size(ad.handles.app_modes_val));
set(ad.handles.app_pc_slider, 'Value', 0);
setappdata(0, 'ModelViewerData', ad);
plotModel;
doPlotModes(ad.handles.applevels, ad.handles.app_modes_val, 'app');
if val<=length(ad.handles.app_modes_val)
    highlightMode(val, ad.handles.applevels);
end
%%%%%
%
%
%%%%%
function doWalkCurrentShapePC(walkcurrbtn, evd) %#ok<INUSD>
ad =getappdata(0, 'ModelViewerData');
slider = ad.handles.shape_pc_slider;
minval = get(slider, 'Min');
maxval = get(slider, 'Max');
step = get(slider, 'SliderStep');
step = step(2);
direction = 1;
val = 0;
while get(walkcurrbtn, 'value') == 1
    ad =getappdata(0, 'ModelViewerData');
    set(slider, 'Value', val);
    val = val + direction*step;
    if val>=maxval  || val<=minval
        direction = direction*-1;
        val = val + direction*step;
    end
    ad.handles.shape_pc_slider = slider;
    setappdata(0, 'ModelViewerData', ad);
    doUpdateShapeMode(ad.handles.shape_pc_slider);
end
return;
%%%%%
%
%
%%%%%
function doWalkCurrentAppPC(walkcurrbtn, evd) %#ok<INUSD>
ad =getappdata(0, 'ModelViewerData');
slider = ad.handles.app_pc_slider;
minval = get(slider, 'Min');
maxval = get(slider, 'Max');
step = get(slider, 'SliderStep');
step = step(2);
direction = 1;
val = 0;
while get(walkcurrbtn, 'value') == 1
    ad =getappdata(0, 'ModelViewerData');
    set(slider, 'Value', val);
    val = val + direction*step;
    if val>=maxval  || val<=minval
        direction = direction*-1;
        val = val + direction*step;
    end
    ad.handles.app_pc_slider = slider;
    setappdata(0, 'ModelViewerData', ad);
    doUpdateAppMode(ad.handles.app_pc_slider);
end
return;
%%%%%
%
%
%%%%%
function doWalkAllShapePC(walkallbtn, evd) %#ok<INUSD>
ad =getappdata(0, 'ModelViewerData');
slider = ad.handles.shape_pc_slider;
minval = get(slider, 'Min');
maxval = get(slider, 'Max');
step = get(slider, 'SliderStep');
step = step(2);
direction = 1;
val = 0;
num_modes = length(ad.handles.shape_modes_val);
walksteps = [0:step:maxval, maxval:-step:minval, minval:step:0];
broken = 0;
for i=ad.currentShapeMode:num_modes
    if get(walkallbtn, 'value')==0
        broken = 1;
        break
    end
    ad.currentShapeMode = i;
    setappdata(0, 'ModelViewerData', ad);
    for ii=walksteps
        if get(walkallbtn, 'value')==0
            break
        end
        ad =getappdata(0, 'ModelViewerData');
        set(slider, 'Value', ii);
        val = val + direction*step;
        if val>=maxval  || val<=minval
            direction = direction*-1;
            val = val + direction*step;
        end
        ad.handles.shape_pc_slider = slider;
        setappdata(0, 'ModelViewerData', ad);
        doUpdateShapeMode(ad.handles.shape_pc_slider);
    end
    ad.handles.shape_modes_val = zeros(size(ad.handles.shape_modes_val));
    setappdata(0, 'ModelViewerData', ad);
end
if broken==0
    set(walkallbtn, 'Value', 0);
end
return;
%%%%%
%
%
%%%%%
function doWalkAllAppPC(walkallbtn, evd) %#ok<INUSD>
ad =getappdata(0, 'ModelViewerData');
slider = ad.handles.app_pc_slider;
minval = get(slider, 'Min');
maxval = get(slider, 'Max');
step = get(slider, 'SliderStep');
step = step(2);
direction = 1;
val = 0;
num_modes = length(ad.handles.app_modes_val);
walksteps = [0:step:maxval, maxval:-step:minval, minval:step:0];
broken = 0;
for i=1:num_modes
    if get(walkallbtn, 'value')==0
        broken = 1;
        break
    end
    ad.currentAppMode = i;
    setappdata(0, 'ModelViewerData', ad);
    for ii=walksteps
        if get(walkallbtn, 'value')==0
            break
        end
        ad =getappdata(0, 'ModelViewerData');
        set(slider, 'Value', ii);
        val = val + direction*step;
        if val>=maxval  || val<=minval
            direction = direction*-1;
            val = val + direction*step;
        end
        ad.handles.app_pc_slider = slider;
        setappdata(0, 'ModelViewerData', ad);
        doUpdateAppMode(ad.handles.app_pc_slider);
    end
    ad.handles.app_modes_val = zeros(size(ad.handles.app_modes_val));
    setappdata(0, 'ModelViewerData', ad);
end
if broken==0
    set(walkallbtn, 'Value', 0);
end
return;
%%%%%
%
%
%%%%%
function doUpdateShapeMode(slider, evd) %#ok<INUSD>
ad =getappdata(0, 'ModelViewerData');
val = ad.currentShapeMode;
if isempty(val) || (val>length(ad.handles.shape_modes_val))
    return;
end
ad.handles.shape_modes_val(val) = get(slider, 'Value');
set(ad.handles.shape_pc_num_txt, 'String', num2str(val));
set(ad.handles.shape_pc_value_txt, 'String', num2str(ad.handles.shape_modes_val(val)));
setappdata(0,'ModelViewerData',ad);
updatePlot;
doPlotModes(ad.handles.shapelevels, ad.handles.shape_modes_val, 'shape');
if val<=length(ad.handles.shape_modes_val)
    highlightMode(val, ad.handles.shapelevels);
end

%%%%%
%
%
%%%%%
function doUpdateAppMode(slider, evd) %#ok<INUSD>
ad =getappdata(0, 'ModelViewerData');
val = ad.currentAppMode;
if isempty(val) || (val>length(ad.handles.app_modes_val))
    return;
end
ad.handles.app_modes_val(val) = get(slider, 'Value');
set(ad.handles.app_pc_num_txt, 'String', num2str(val));
set(ad.handles.app_pc_value_txt, 'String', num2str(ad.handles.app_modes_val(val)));
setappdata(0,'ModelViewerData',ad);
plotModel;
doPlotModes(ad.handles.applevels, ad.handles.app_modes_val, 'app');
if val<=length(ad.handles.app_modes_val)
    highlightMode(val, ad.handles.applevels);
end
%%%%%
%
%
%%%%%
function plotModel
ad =getappdata(0, 'ModelViewerData');
drawnow;
cla(ad.handles.mainAxes);
ad = plotReferenceImage(ad);

hold(ad.handles.mainAxes, 'on');
if get(ad.handles.show_app_model_chk, 'Value')
    ad = plotAppearanceModel(ad);
    hold(ad.handles.mainAxes, 'on');    
end
hold(ad.handles.mainAxes, 'on');
ad = plotEdges(ad);
ad = plotShapeModel(ad);
hold(ad.handles.mainAxes, 'off');
togglePlotView;
axis(ad.handles.mainAxes, 'equal');
axis(ad.handles.mainAxes, [ad.xrange(1) ad.xrange(2), ad.yrange(1) ad.yrange(2)]);
set(ad.handles.mainAxes, 'Color', [0 0 0]);
setappdata(0, 'ModelViewerData', ad);
title(ad.handles.mainAxes, '', 'Interpreter', 'none');
drawnow;
%%%%%
%
%
%%%%%
function updatePlot
ad =getappdata(0, 'ModelViewerData');
ad = updateReferenceImage(ad);
if get(ad.handles.show_app_model_chk, 'Value')
    ad = updateAppearanceModel(ad);
end
ad = updateEdges(ad);
ad = updateShapeModel(ad);
togglePlotView;
axis(ad.handles.mainAxes, 'equal');
% Begin change by js, 070705
% Begin original code
 
% axis(ad.handles.mainAxes, [ad.xrange(1) ad.xrange(2), ad.yrange(1) ad.yrange(2)]);
 
% End original code
% Begin replacement code
 
switch ad.dim
    case 3
        axis(ad.handles.mainAxes, [ad.xrange(1) ad.xrange(2), ad.yrange(1) ad.yrange(2), ad.zrange(1) ad.zrange(2)]);
    case 2
        axis(ad.handles.mainAxes, [ad.xrange(1) ad.xrange(2), ad.yrange(1) ad.yrange(2)]);
end
 
% End replacement code
% End change by js, 070705
set(ad.handles.mainAxes, 'Color', [0 0 0]);
setappdata(0, 'ModelViewerData', ad);
title(ad.handles.mainAxes, '', 'Interpreter', 'none');
drawnow;
%%%%%
%
%
%%%%%
% function setShapeSliders(b)
% ad =getappdata(0, 'ModelViewerData');
% setappdata(0, 'ModelViewerData', ad);
%%%%%
%
%
%%%%%
function ad = plotShapeModel(ad)
if isempty(ad.pdm)
    return;
end
if get(ad.handles.show_landmarks_chk, 'Value')==0
    return;
end
pts = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val, ad.dim);
X = pts';
X = X(:);

% Calculate the Euclidean Distance from the formed shape to the mean.
diff = norm(ad.pdm.Xm - X(:));
set(ad.handles.eucdisttxt, 'String', num2str(diff));
if ~isempty(ad.segmentPoint)
    pts = splitPoints(pts, ad.segmentPoint, ad.offsets);
end
ad.currentShapeModel = pts;
ad = updateAxisLimits(ad, pts);
switch ad.dim
    case 3
        ad.data.shape_pts_handle = plot3(ad.handles.mainAxes, pts(:,1), pts(:,2), pts(:,3), 'o', 'MarkerFaceColor', 'y', 'MarkerEdgeColor', 'r');
    case 2
        ad.data.shape_pts_handle = plot(ad.handles.mainAxes, pts(:,1), pts(:,2), 'o', 'MarkerFaceColor', 'y', 'MarkerEdgeColor', 'r');
end

%%%%%
%
%
%%%%%
function ad = updateShapeModel(ad)
if isempty(ad.pdm)
    return;
end
if get(ad.handles.show_landmarks_chk, 'Value')==0
    return;
end
pts = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val, ad.dim);
X = pts';
X = X(:);
% Calculate the Euclidean Distance from the formed shape to the mean.
diff = norm(ad.pdm.Xm - X(:));
set(ad.handles.eucdisttxt, 'String', num2str(diff));
if ~isempty(ad.segmentPoint)
    pts = splitPoints(pts, ad.segmentPoint, ad.offsets);
end
ad.currentShapeModel = pts;
ad = updateAxisLimits(ad, pts);
set(ad.data.shape_pts_handle, 'XData', pts(:,1), 'YData', pts(:,2));
%%%%%
%
%
%%%%%
function ad = plotEdges(ad)
if isempty(ad.pdm)
    return;
end
if get(ad.handles.edges_chk, 'Value')==0
    return;
end
pts = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val, ad.dim);
if ~isempty(ad.segmentPoint)
    pts = splitPoints(pts, ad.segmentPoint, ad.offsets);
end
if max(ad.edges(:))<=size(pts,1)
    if ad.dim==3
        for i=1:size(ad.edges,1)
            ad.data.edgehandle(i) = plot3(ad.handles.mainAxes, [pts(ad.edges(i,1),1), pts(ad.edges(i,2), 1)],...
                [pts(ad.edges(i,1),2), pts(ad.edges(i,2),2)], [pts(ad.edges(i,1),3), pts(ad.edges(i,2),3)], ...
                '-', 'Color', 'w', 'LineWidth', 3, 'MarkerFaceColor', 'y', 'MarkerEdgeColor', 'r');
           % ad.data.edgehandle(i) = plot(ad.handles.mainAxes, [pts(ad.edges(i,1),1), pts(ad.edges(i,2), 1)], [pts(ad.edges(i,1),2), pts(ad.edges(i,2),2)], '-', 'Color', 'w', 'LineWidth', 3, 'MarkerFaceColor', 'y', 'MarkerEdgeColor', 'r');
        end
    else

        for i=1:size(ad.edges,1)
            ad.data.edgehandle(i) = plot(ad.handles.mainAxes, [pts(ad.edges(i,1),1), pts(ad.edges(i,2), 1)], [pts(ad.edges(i,1),2), pts(ad.edges(i,2),2)], '-', 'Color', 'w', 'LineWidth', 3, 'MarkerFaceColor', 'y', 'MarkerEdgeColor', 'r');
        end
    end
end
return
%%%%%
%
%
%%%%%
function ad = updateEdges(ad)
if isempty(ad.pdm)
    return;
end
if get(ad.handles.edges_chk, 'Value')==0
    return;
end
pts = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val, ad.dim);
if ~isempty(ad.segmentPoint)
    pts = splitPoints(pts, ad.segmentPoint, ad.offsets);
end
for i=1:length(ad.data.edgehandle)
    set(ad.data.edgehandle(i), 'XData', [pts(ad.edges(i,1),1), pts(ad.edges(i,2), 1)], 'YData', [pts(ad.edges(i,1),2), pts(ad.edges(i,2),2)]);
end
return

%%%%%
%
%
%%%%%
function model = getCurrShapeModel(pdm, modes_val, dim)
ad =getappdata(0, 'ModelViewerData');
Xm =pdm.Xm;
P = pdm.P;
b = pdm.b;
model_b = modes_val.*sqrt(b);
X = Xm + P*model_b;
model = reshape(X, dim , length(X)/dim )';
% if we have an appearance model then we should keep all the points
% positive
if isfield(ad, 'options')
    if ~isempty(ad.options.sfam)
        model = model - ones(size(model,1),1)*min(model);
    end
end
return;
%%%%%
%
%
%%%%%
function ad = plotAppearanceModel(ad)    
if ~isfield(ad, 'sfam')
    return;
end
if isempty(ad.sfam)
    return;
end
sz = ad.sfam.siz;
Am = ad.sfam.Am;
P = ad.sfam.P;
b = ad.sfam.b;
ompts = ad.sfam.ompts;
model_b = ad.handles.app_modes_val.*sqrt(b);
A = Am + P*model_b;
I = reshape(A, [sz(2) sz(1) sz(3)]);
shape_model = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val, ad.dim);
if get(ad.handles.tps_chk, 'Value')
    fprintf('Calculating TPS warp...');
    %ad.currentImage = uint8(tpswarp_c(I,ompts,shape_model, 1.1));
    ad.currentImage = TPS_im_warp(I,shape_model,ompts);
    fprintf('done.\n');
else

    ad.currentImage = uint8(triwarp(I,ompts,shape_model));
end
ad.data.appearancehandle = imagesc(ad.currentImage, 'Parent', ad.handles.mainAxes);
return;
%%%%%
%
%
%%%%%
function ad = plotReferenceImage(ad)
if isempty(ad.options.ref_image)
    return;
end
I = ad.options.ref_image.image;
ompts = ad.options.ref_image.pts;
ompts = reshape(ompts, ad.dim , length(ompts)/ad.dim )';
shape_model = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val, ad.dim);
ad.currentImage = uint8(warp_to_mean_shape(ompts', shape_model,I));
%ad.currentImage = uint8(triwarp(I,ompts,shape_model));
ad.data.appearancehandle = imagesc(ad.currentImage, 'Parent', ad.handles.mainAxes);
return;
%%%%%
%
%
%%%%%
function ad = updateAppearanceModel(ad)
if ~isfield(ad, 'sfam')
    return;
end
if isempty(ad.sfam)
    return;
end
sz = ad.sfam.siz;
Am = ad.sfam.Am;
P = ad.sfam.P;
b = ad.sfam.b;
ompts = ad.sfam.ompts;
model_b = ad.handles.app_modes_val.*sqrt(b);
A = Am + P*model_b;
I = reshape(A, [sz(2) sz(1) sz(3)]);
shape_model = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val, ad.dim);
if get(ad.handles.tps_chk, 'Value')
    fprintf('Calculating TPS warp...');
    %ad.currentImage = uint8(tpswarp_c(I,ompts,shape_model, 1.1));
    ad.currentImage = TPS_im_warp(I,shape_model,ompts);
    fprintf('done.\n');
else
    ad.currentImage = uint8(triwarp(I,ompts,shape_model));
end
set(ad.data.appearancehandle, 'CData', ad.currentImage);
return;
%%%%%
%
%
%%%%%
function ad = updateReferenceImage(ad)
if isempty(ad.options.ref_image)
    return;
end
I = ad.options.ref_image.image;
ompts = ad.options.ref_image.pts;
ompts = reshape(ompts, ad.dim , length(ompts)/ad.dim )';
shape_model = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val, ad.dim);
ad.currentImage = uint8(warp_to_mean_shape(ompts', shape_model,I));
%ad.currentImage = uint8(triwarp(I,ompts,shape_model));
set(ad.data.appearancehandle, 'CData', ad.currentImage);
return;
%%%%%
%
%
%%%%%
% function [pts] = collectPoints(flipind, pts)
% for i=1:length(flipind)
%     p = flipind(i).point;
%     n = flipind(i).neighbours;
%     v1 = (pts(n(1),:) - pts(n(2),:))';
%     v2 = (pts(p,:) - pts(n(2),:))';
%     if (v1'*v1)>0
%         pts(p,:) = pts(n(2),:) + ((v1'*v2)/(v1'*v1))*v1';
%     end
% end
% return

% function showFlipped(ipts, opts)
% tri = delaunay(ipts(:,1), ipts(:,2),{'QJ','QJ','QJ'});
% flipped_tri_ind = find_flipped_triangles(tri, ipts, opts);
% 
% figure(1); clf; hold on;
% triplot(tri, opts(:,1), opts(:,2)); axis image ij;
% 
% for i=1:length(flipped_tri_ind);
%     ind = flipped_tri_ind(i);
%     pts = opts(tri(ind,:),:);
%     p = patch(pts(:,1), pts(:,2), 'r');
% end
% switch ad.dim
%     case 3
%         plot3(opts(:,1), opts(:,2), opts(:,3), 'rs');
%     case 2
%         plot(opts(:,1), opts(:,2), 'rs');
% end
% return;
%%%%%
%
%
%%%%%
function doSlideModes(slider, evd, axes_h) %#ok<INUSL>
ad =getappdata(0, 'ModelViewerData');
YLIM = -get(slider,'value')+sort([0 ad.dy])-.5;
set(axes_h,'ylim',YLIM);
return;
%%%%%
%
%
%%%%%
function showShapeModeInfo(mode)
ad =getappdata(0, 'ModelViewerData');
if isempty(mode) || (mode>length(ad.handles.shape_modes_val))
    return;
end
set(ad.handles.shape_pc_num_txt, 'String', num2str(mode));
set(ad.handles.shape_pc_value_txt, 'String', num2str(ad.handles.shape_modes_val(mode)));
val = max(abs(ad.handles.shape_modes_val));
if val>2
    set(ad.handles.shape_pc_slider, 'Max', val, 'Min', -val);
else
    set(ad.handles.shape_pc_slider, 'Max', 2, 'Min', -2);
end
set(ad.handles.shape_pc_slider, 'Value', ad.handles.shape_modes_val(mode));
setappdata(0,'ModelViewerData',ad);
return;
%%%%%
%
%
%%%%%
function showAppModeInfo(mode)
ad =getappdata(0, 'ModelViewerData');
if isempty(mode) || (mode>length(ad.handles.app_modes_val))
    return;
end
set(ad.handles.app_pc_num_txt, 'String', num2str(mode));
set(ad.handles.app_pc_value_txt, 'String', num2str(ad.handles.app_modes_val(mode)));
set(ad.handles.app_pc_slider, 'Value', ad.handles.app_modes_val(mode));
setappdata(0,'ModelViewerData',ad);
return;
%%%%%
%
%
%%%%%
function doSelectMode(axis_h, evd, slider_h, str) %#ok<INUSL,INUSL>
ad =getappdata(0, 'ModelViewerData');
p = get(axis_h, 'CurrentPoint');
y = round(p(1,2));
switch str
    case 'shape'
        modes_val = ad.handles.shape_modes_val;
        ad.currentShapeMode = y;
        showShapeModeInfo(y);
        doPlotModes(axis_h, modes_val, 'shape');

    case 'app'
        modes_val = ad.handles.app_modes_val;
        ad.currentAppMode = y;
        showAppModeInfo(y);
        doPlotModes(axis_h, modes_val, 'app');
    otherwise
        disp('Unknown model type');
end
hold(axis_h, 'on');
if y<=length(modes_val)
    highlightMode(y, axis_h);
end
setappdata(0, 'ModelViewerData', ad);
return;
%%%%%
%
%
%%%%%
function doPlotModes(axis_h, modes_val, type)
ad =getappdata(0, 'ModelViewerData');
width = 6;
height = 10;
xoffset = 4;
switch type
    case 'shape'
        walkerlim = ad.shapewalkerlim;
    case 'app'
        walkerlim = ad.appwalkerlim;
end
modes_val = real(modes_val);
YLIM = get(axis_h, 'YLim');
cla(axis_h);
hold(axis_h, 'on');
N = length(modes_val);
maxval = max(abs(modes_val));
%if maxval>2
    %xpos = modes_val./maxval;
%    set(ad.handles.shape_range, 'String', num2str(maxval));
%else
%    set(ad.handles.shape_range, 'String', num2str(2));
    xpos = (modes_val./walkerlim);
%end
for i=1:N
    str = sprintf('%s', 'PC ', num2str(i));
    text(1, i, str,  'Color', 'k', 'HitTest', 'off', 'Clipping', 'on', 'Parent', axis_h, 'FontSize', 8);
    if modes_val(i)>0
        color = 'r';
    else
        color = 'b';
    end
    X = [xoffset xoffset+xpos(i)];
    plot(axis_h, X,[i i], '-', 'Color', color, 'LineWidth', 4);
end
axis(axis_h, [1-.2 width .5 height+.5], 'ij');
set(axis_h, 'XMinorGrid', 'on');
set(axis_h, 'XTick', [], 'YTick', [], 'Clipping', 'on', 'Box', 'on');
set(axis_h, 'YLim', YLIM);
%%%%%
%
%
%%%%%
function highlightMode(mode, axis_h)
width = 2.5;
X = [1 width-.2 width-.2 1];
Y = [mode-.4 mode-.4 mode+.4 mode+.4];
C = [0.1412    0.3137    1];
%patch(X, Y, C, 'FaceAlpha', .1, 'Parent', axis_h, 'EdgeColor', 'k', 'MarkerSize', .1, 'LineWidth', 0.01, 'LineStyle', ':', 'Clipping', 'on');
patch(X, Y, C, 'Parent', axis_h, 'Clipping', 'on');
str = sprintf('%s', 'PC ', num2str(mode));
text(1, mode, str, 'Clipping', 'on', 'Parent', axis_h, 'FontSize', 8, 'HitTest', 'off', 'Color', 'w');
return;
%%%%%%%%%%%%%%%%%%%%
%
% Disable Appearance buttons
%
%%%%%%%%%%%%%%%%%%%%
function handles = toggleAppearanceGUI(handles, enablestr)
set(handles.applevels, 'Visible', enablestr);
set(handles.appslider, 'Enable', enablestr);
set(handles.app_pc_slider, 'Enable', enablestr);
set(handles.walk_curr_app_pc_btn, 'Enable', enablestr);
set(handles.walk_all_app_pc_btn, 'Enable', enablestr);
set(handles.reset_app_btn, 'Enable', enablestr);
set(handles.reset_all_app_btn, 'Enable', enablestr);
set(handles.app_range, 'Enable', enablestr);


%%%%%%%%%%%%%%%%%%%%
%
% SetGUILevel
%
%%%%%%%%%%%%%%%%%%%%
function handles = SetGUILevel(handles, level)
if level==1
    set(handles.gen_opts_panel, 'Visible', 'off');
    set(handles.add_rand_axis_btn, 'Visible', 'off');
    set(handles.reload_model_btn, 'Visible', 'off'); 
end
%%%%
%
%%%%
function ad = savePDM(ad)
queststr = sprintf('Do you want to save the changes to your shape model?');
savebtn = questdlg(queststr, 'Save shape model', 'Yes', 'No','Yes');
if strcmpi(savebtn, 'yes')
    pdm = ad.pdm;
    save(ad.options.pdm, 'pdm');
    fprintf('Saved pdm to %s\n', ad.options.pdm);
end
return;
%%%
%
%%%
function ad = manualInitialization(ad)
%directoryname = uigetdir(pwd, 'Please choose the directory where your model resides');
[filename, directoryname] = uigetfile('*.mat', 'Pick PDM model to open');
if isequal(filename,0) || isequal(directoryname,0)
    disp('User pressed cancel');
    return
end

pdmname = [directoryname, filesep, filename];
sfamname = [directoryname, filesep, 'mod_sfam.mat'];
aamfile = [directoryname, filesep, 'model.aam_dat'];
if exist(pdmname, 'file')
    ad.options.pdm = pdmname;
end
if exist(sfamname, 'file')
    ad.options.sfam = sfamname;
end
if exist(aamfile, 'file')
    aam = load(aamfile, '-mat');
    aam = aam.aam;
    template = get(aam, 'PointModelTemplate');
    edges = get(template, 'loops');
    if ~isempty(edges)
        ad.edges = edges{1};
    end
    templatename = get(template, 'name');
    [temppath, templatename, ext, vers] = fileparts(templatename);
    pmdir = ['PointModels', filesep, templatename];
  %  if exist(pmdir, 'dir')
   % else
   %     ad.options.PMdirectory = uigetdir(pwd, 'Please choose the directory where your point models live');
   % end
%else
%    ad.options.PMdirectory = uigetdir(pwd, 'Please choose the directory where your point models live');
end
%%%
%
%%%
function doImportRefImage(refimagemenu, evd) %#ok<INUSD>
ad =getappdata(0, 'ModelViewerData');
imdir = [ad.options.ImageDirectory];
pmdir = [ad.options.PMdirectory];
if isempty(pmdir)
    pmdir = uigetdir(pwd, 'Choose your point model directory');
    ad.options.PMdirectory = pmdir;
end
[filename, pathname] = uigetfile('*.jpg', 'Pick a Reference Image', [imdir, filesep]);
if isequal(filename,0) || isequal(pathname,0)
    disp('User pressed cancel');
    return;
else
    ref_image.image = imread([pathname, filesep, filename]);
     ref_image.image = double(ref_image.image);
%     ref_image.image = ref_image.image./max(ref_image.image(:));
    [pathname, filename, ext, vers] = fileparts(filename);
    pmfilename = [pmdir, filesep, filename, '_pm.mat'];
    if ~exist(pmfilename, 'file')
        uiwait(msgbox('There is no point model to match this image. Please try again.','File not found','modal'));
        return;
    end
    pts = load(pmfilename);
    ref_image.pts = pts.pts;
    disp(['User selected ', fullfile(pathname, filename)])
end
ad.options.ref_image = ref_image;
setappdata(0, 'ModelViewerData', ad);
plotModel;
return;
%%%
%
%%%
function customtoolbar(figh)
hA = findall(figh);
openfileh=findobj(hA,'TooltipString','Open File');
%set(openfileh, 'ClickedCallback', {@doOpenGrid});
%set(openfileh, 'TooltipString', 'Open Grid File');
newgridh=findobj(hA,'TooltipString','New Figure');
%set(newgridh, 'ClickedCallback', {@doNewGrid});
%set(newgridh, 'TooltipString', 'New Grid');
saveh=findobj(hA,'TooltipString','Save Figure');
%set(saveh, 'ClickedCallback', {@doSaveGrid});
%set(saveh, 'TooltipString', 'Save Grid');
roth=findobj(hA,'TooltipString','Rotate 3D');
datah=findobj(hA,'TooltipString','Data Cursor');
colbarh=findobj(hA,'TooltipString','Insert Colorbar');
legh=findobj(hA,'TooltipString','Insert Legend');
hideh=findobj(hA,'TooltipString','Hide Plot Tools');
showh=findobj(hA,'TooltipString','Show Plot Tools and Dock Figure');
delete([newgridh, saveh, openfileh, datah, colbarh, legh, hideh, showh]); 