function [output] = templateBuilder3DTool(varargin)
% templateBuilder3DTool M-file for templateBuilder3DTool.fig
%
% Description:
%
% 
% Author: Johann Strasser
% Date: 070312

error(nargchk(0, inf, nargin));

if mod(length(varargin), 2) ~= 0
    % input did not come in parameter/value pairs
    error('Arguments have to come in parameter/value pairs.');
end

% Initialise default application data structure to show all the fields.
ad.guiHandles = [];
ad.projectDirectory = '';
ad.aam = [];
ad.output = [];

% Process input arguments
for i=1:2:length(varargin)
    switch lower(varargin{i})
%         case 'aam'
%             ad.aam = varargin{i + 1};
        case 'projectdirectory'
            ad.projectDirectory = varargin{i + 1};
        otherwise
            error(['Unknown parameter name passed to templateBuilder3DTool. ', ...
                'Parameter name: ',  varargin{i}]);
    end
end

% Check if required application data was set, if not set default values
% and issue warning
if isempty(ad.projectDirectory)
   ad.projectDirectory = '.';
   warning(sprintf(['TemplateBuilder3DTool: Parameter ''projectDirectory'' not provided. \n', ...
       'Using current directory (.) as default']));
end

% if isempty(ad.aam)
%    ad.aam = AAM;
%    warning(sprintf(['TemplateBuilder3DTool: Parameter ''aam'' not provided. \n', ...
%        'Using empty AAM object as default']));
% end

% Open figure file
fig = openfig(mfilename, 'new');

% Get handles
handles = guihandles(fig);
ad.guiHandles = handles;

% Set the callback functions for all the gui elements
set(fig, 'closeRequestFcn', {@customCloseRequestFcn});
set(ad.guiHandles.editTemplatePshBtn, 'callback', {@editTemplateCallback})
set(ad.guiHandles.newTemplatePshBtn, 'callback', {@newTemplateCallback})
set(ad.guiHandles.viewVolumePshBtn, 'callback', {@viewVolumeCallback})

% Setup GUI elements
setupGUIElements(ad);

% Save the application data
setappdata(0, 'templateBuilder3DToolData', ad);

% Move the gui and show it
movegui(fig, 'center');
set(fig, 'visible', 'on');

% Make sure this is a modal dialog
try
    % Wait until resume is called or figure is deleted
    uiwait(fig);
catch
    if ishandle(fig)
        delete(fig)
    end
end

% display('M-file execution resumed.');

% Set return arguments here. The fact that the figure has been destroyed
% is irrelevant as the appdata is associated with the root handle graphics
% object
ad = getappdata(0, 'templateBuilder3DToolData');
output = ad;
rmappdata(0, 'templateBuilder3DToolData');

function viewVolumeCallback(cbo, event)

% display('viewVolumeCallback executing.');

ad = getappdata(0, 'templateBuilder3DToolData');

% Set the figure invisible, note that the event is queued as uiwait was
% called on the figure handle. Hence call drawnow expose to process events
% causing graphics object updates only.
set(ad.guiHandles.templateBuilder3DToolFig, 'visible', 'off');
drawnow expose; 
viewVolume('path', fullfile(ad.projectDirectory, 'Cropped'));
set(ad.guiHandles.templateBuilder3DToolFig, 'visible', 'on');
setappdata(0, 'templateBuilder3DToolData', ad);

function newTemplateCallback(cbo, event)

display('newTemplateCallback executing.');

ad = getappdata(0, 'templateBuilder3DToolData');
ad = newTemplate(ad);
setappdata(0, 'templateBuilder3DToolData', ad);

function editTemplateCallback(cbo, event)

display('editTemplateCallback executing.');

ad = getappdata(0, 'templateBuilder3DToolData');
ad = editTemplate(ad);
setappdata(0, 'templateBuilder3DToolData', ad);

function customCloseRequestFcn(cbo, event)

% display('customCloseRequestFcn executing.');

% The following is the code in the default close request function
% (closereq)
if isempty(gcbf)
    if length(dbstack) == 1
        warning('MATLAB:customCloseRequestFcn', ...
                'Calling customCloseRequestFcn from the command line is now obsolete, use close instead');
    end
    close force
else
    delete(gcbf);
end

function setupGUIElements(ad)

% Populate the list box
% volumeNames = get(ad.aam, 'imageFilenames');
% set(ad.guiHandles.volumesListBox, 'String', volumeNames);

function ad = editTemplate(ad)

imreadSupportedFileTypes = getImreadSupportedTypesFilterSpec();    

[fileName, path] = uigetfile('.temp_dat', ...
    'Choose template', fullfile(ad.projectDirectory, ...
    'Templates', filesep)); 
    
if isequal(path, 0) && isequal(fileName, 0)
    disp('User selected cancel.'); 
    userCancel = 1;
else
    [fullVolumePath, name, extension, versn] = fileparts(fullfile(path, fileName));
    
     % Load template
    temp = load([path, fileName], '-mat');
    PMTemplate = temp.PMTemplate;
     
    volumeName = get(PMTemplate, 'imageName');
    imageDir = get(PMTemplate, 'imageDir');
    templateName = get(PMTemplate, 'name');
    directory = get(PMTemplate, 'direc');
    vertices = get(PMTemplate, 'pts');
    primaries = get(PMTemplate, 'primaries');
    loops = get(PMTemplate, 'loops');
    lineIndices = get(PMTemplate, 'lineIndices');
    vertexColours = get(PMTemplate, 'vertexColours');
    lineColours = get(PMTemplate, 'lineColours');

    % Load volume
    fullVolumePath = fullfile(ad.projectDirectory, imageDir, volumeName);
    volume = loadVolumeFromSlicesDir(fullVolumePath);
   
    % Initialise the transform structure for the template
    t = getHTTransform();
    t.name = 'tVolumeTransform';
    
    % Initialise volume structure for the template
    v = getHTVolume();
    v.name = 'tVolume';
    v.parentTransform = 'tVolumeTransform';
    v.volume = volume;
    v.planes = 256;
   
    ils = getHTIndexedLineSet();
    ils.name = 'tIndexedLineSet';
    ils.parentTransform = 'tVolumeTransform';
    ils.vertices = vertices; 
    ils.lineIndices = lineIndices; 
    ils.interactionMode = 'edit';
    ils.vertexColours = vertexColours;
    ils.lineColours = lineColours;
    ils.vertexStiffness = 0.5;
    ils.lineStiffness = 0.5;
   
    cp = getHTClipPlane();
    cp.name = 'clipPlane1';
    cp.parentTransform = 'tVolumeTransform';
    cp.point = [0, -0.05, 0];
    cp.normal = [0, 1, 0];
    cp.interactionMode = 'edit';
    
    % Set the figure invisible, note that the event is queued as uiwait was
    % called on the figure handle. Hence call drawnow expose to process events
    % causing graphics object updates only.
    set(ad.guiHandles.templateBuilder3DToolFig, 'visible', 'off');
    drawnow expose; 
    
    % Run hapticTool
%     [tOut, vOut, pOut, ilsOut, cpOut] =...
%             hapticTool(t, v, [], ils, cp);
        
    [tOut, vOut, pOut, ilsOut, cpOut] =...
            hapticTool(t, v, [], ils, []);

    set(ad.guiHandles.templateBuilder3DToolFig, 'visible', 'on');

    % Get the volume directory name
    [starts, ends, extents, matches, tokens, names] = ...
        regexpi(fullVolumePath, ['[^', '\', filesep, ']+']);
    		 vertices = ilsOut.vertices;
             
    primaries = [];
    loops = [];

    % Scan through the line indices field and convert to primaries/loops
    % convention. Conversion rules are as follows (simple by choice of 
    % native data structure):
    % Polyline end vertices are primaries
    % Each polyline segment is a loop
    for i = 1:length(ilsOut.lineIndices)
        if ilsOut.lineIndices(i) == -1
            lastPrimary = [];
            nextPrimary = [];
            if i ~= length(ilsOut.lineIndices)
                nextPrimary = ilsOut.lineIndices(i + 1);
            end

            if i ~= 1
                lastPrimary = ilsOut.lineIndices(i - 1);
            end

            % Avoid double recording of a primary point if
            % polylines are joint
            if nextPrimary == lastPrimary
                primaries = [primaries, nextPrimary];
            else
                if ~isempty(nextPrimary)
                    primaries = [primaries, nextPrimary];
                end

                if ~isempty(lastPrimary)
                    primaries = [primaries, lastPrimary];
                end
            end
        else
            if ilsOut.lineIndices(i + 1) ~= -1
                loops = [loops; ...
                    [ilsOut.lineIndices(i), ilsOut.lineIndices(i + 1)]];
            end
        end
    end     

           
    % Set member variables of PMTemplate object
    loopsCell = {};
    loopsCell{1, 1} = loops + 1;
    PMTemplate = PointModelTemplate;
    PMTemplate = set(PMTemplate, 'imageName', volumeName);
    PMTemplate = set(PMTemplate, 'imageDir', 'Templates');
    PMTemplate = set(PMTemplate, 'name', templateName);
    PMTemplate = set(PMTemplate, 'direc', 'Templates');
    PMTemplate = set(PMTemplate, 'pts', vertices);
    PMTemplate = set(PMTemplate, 'primaries', primaries);
    PMTemplate = set(PMTemplate, 'loops', loopsCell);
    PMTemplate = set(PMTemplate, 'lineIndices', ilsOut.lineIndices);
    PMTemplate = set(PMTemplate, 'vertexColours', ilsOut.vertexColours);
    PMTemplate = set(PMTemplate, 'lineColours', ilsOut.lineColours);

%     volumeName = char(matches(length(matches)));
    
    % Name and path to be requested from and/or confirmed by the user
%     defaultTemplatePath = fullVolumePath;
    defaultTemplateName = templateName;
    
    % Save the template if desired
    button = questdlg('Save template?', 'Save', 'Yes', 'Save as', 'No', 'Yes');
    
    if isequal(button, 'Save as')        
        [templateName, templatePath] = uiputfile('.temp_dat', ...
            'Save template', fullfile(ad.projectDirectory, ...
             'Templates', defaultTemplateName)); 
        
         if isequal(templatePath, 0) && isequal(templateName, 0)
            disp('User selected cancel.'); 
            userCancel = 1;
         else              
            save(fullfile(templatePath, templateName), 'PMTemplate');
         end
    elseif isequal(button, 'Yes')
        templatePath = fullfile(ad.projectDirectory, 'Templates');
        save(fullfile(templatePath, templateName), 'PMTemplate'); 
    end
end

function ad = newTemplate(ad)

imreadSupportedFileTypes = getImreadSupportedTypesFilterSpec();    

[fileName, path] = uigetfile(imreadSupportedFileTypes, ...
    'Choose template volume', fullfile(ad.projectDirectory, ...
    'Templates', filesep)); 
    
if isequal(path, 0) && isequal(fileName, 0)
    disp('User selected cancel.'); 
    userCancel = 1;
else
    [fullVolumePath, name, extension, versn] = fileparts(fullfile(path, fileName));
   
    % Load volume
    volume = loadVolumeFromSlicesDirTyped(fullVolumePath, extension);
    
    % Initialise the transform structure for the template
    t = getHTTransform();
    t.name = 'tVolumeTransform';
    
    % Initialise volume structure for the template
    v = getHTVolume();
    v.name = 'tVolume';
    v.parentTransform = 'tVolumeTransform';
    v.volume = volume;
    v.planes = 256;
   
    ils = getHTIndexedLineSet();
    ils.name = 'tIndexedLineSet';
    ils.parentTransform = 'tVolumeTransform';
    ils.vertices = []; 
    ils.lineIndices = []; 
    ils.interactionMode = 'edit';
    ils.vertexColours = []';
    ils.vertexStiffness = 0.5;
    ils.lineStiffness = 0.5;
   
    cp = getHTClipPlane();
    cp.name = 'clipPlane1';
    cp.parentTransform = 'tVolumeTransform';
    cp.point = [0, -0.05, 0];
    cp.normal = [0, 1, 0];
    cp.interactionMode = 'edit';
    
    % Set the figure invisible, note that the event is queued as uiwait was
    % called on the figure handle. Hence call drawnow expose to process events
    % causing graphics object updates only.
    set(ad.guiHandles.templateBuilder3DToolFig, 'visible', 'off');
    drawnow expose; 
    
    % Run hapticTool
%     [tOut, vOut, pOut, ilsOut, cpOut] =...
%             hapticTool(t, v, [], ils, cp);

   [tOut, vOut, pOut, ilsOut, cpOut] =...
            hapticTool(t, v, [], ils, []);
        
    set(ad.guiHandles.templateBuilder3DToolFig, 'visible', 'on');

    % Get the volume directory name
    [starts, ends, extents, matches, tokens, names] = ...
        regexpi(fullVolumePath, ['[^', '\', filesep, ']+']);
    
    volumeName = char(matches(length(matches)));
    
    % Name and path to be requested from and/or confirmed by the user
    defaultTemplateName = [volumeName, '_3D.temp_dat'];
    
    % Save the template if desired
    button = questdlg('Save new template?', 'Save', 'Yes', 'No', 'Yes');
    
    if isequal(button, 'Yes')        
        [templateName, templatePath] = uiputfile('.temp_dat', ...
            'Save template', fullfile(ad.projectDirectory, ...
             'Templates', defaultTemplateName)); 
        
         if isequal(templatePath, 0) && isequal(templateName, 0)
            disp('User selected cancel.'); 
            userCancel = 1;
         else              
            vertices = ilsOut.vertices;
            primaries = [];
            loops = [];

            % Scan through the line indices field and convert to primaries/loops
            % convention. Conversion rules are as follows (simple by choice of 
            % native data structure):
            % Polyline end vertices are primaries
            % Each polyline segment is a loop
            for i = 1:length(ilsOut.lineIndices)
                if ilsOut.lineIndices(i) == -1
                    lastPrimary = [];
                    nextPrimary = [];
                    if i ~= length(ilsOut.lineIndices)
                        nextPrimary = ilsOut.lineIndices(i + 1);
                    end

                    if i ~= 1
                        lastPrimary = ilsOut.lineIndices(i - 1);
                    end
                    
                    % Avoid double recording of a primary point if
                    % polylines are joint
                    if nextPrimary == lastPrimary
                        primaries = [primaries, nextPrimary];
                    else
                        if ~isempty(nextPrimary)
                            primaries = [primaries, nextPrimary];
                        end
                        
                        if ~isempty(lastPrimary)
                            primaries = [primaries, lastPrimary];
                        end
                    end
                else
                    if ilsOut.lineIndices(i + 1) ~= -1
                        loops = [loops; ...
                            [ilsOut.lineIndices(i), ilsOut.lineIndices(i + 1)]];
                    end
                end
            end

            loopsCell = {};
            loopsCell{1, 1} = loops + 1;
            
            % Set member variables of PMTemplate object
            PMTemplate = PointModelTemplate;
            PMTemplate = set(PMTemplate, 'imageName', volumeName);
            PMTemplate = set(PMTemplate, 'imageDir', 'Templates');
            PMTemplate = set(PMTemplate, 'name', templateName);
            PMTemplate = set(PMTemplate, 'direc', 'Templates');
            PMTemplate = set(PMTemplate, 'pts', vertices);
            PMTemplate = set(PMTemplate, 'primaries', primaries);
            PMTemplate = set(PMTemplate, 'loops', loopsCell);
            PMTemplate = set(PMTemplate, 'lineIndices', ilsOut.lineIndices);
            PMTemplate = set(PMTemplate, 'vertexColours', ilsOut.vertexColours);
            PMTemplate = set(PMTemplate, 'lineColours', ilsOut.lineColours);

            save(fullfile(templatePath, templateName), 'PMTemplate');
         end
    end
end
