# Copyright 2006, Karljohan Lundin
#

"""This file provides extra functionality to the VHTK package. For
more information read the comments for the included classes.


 This file is part of Volume Haptics Toolkit.

 Volume Haptics Toolkit is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 Volume Haptics Toolkit is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with Volume Haptics Toolkit; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""

from H3D import *
from H3DInterface import *

GRAB_DISTANCE  = 0.01
SMALL_DISTANCE = 0.005

refs = references.getValue()

# Clipped group
groupA = refs[0]

# Unclipped group for icons
if len(refs) > 1:
  groupB = refs[1]
else:
  groupB = groupA

# LocalInfo for the groups
if len(refs) > 2:
  info = refs[2]
else:
  info = None

# An icon shape
if len(refs) > 3:
  icon = refs[3]
else:
  icon = createX3DNodeFromString( """
   <Group>
   <MagneticPointEffect springConstant="200" startDistance="0.01" escapeDistance="0.01" />
    <Shape>
     <Appearance DEF="APP">
      <Material diffuseColor=".4 .3 .8"/>
     </Appearance>
     <Sphere radius="0.005"/>
    </Shape>
    <Transform
      translation="0 0 0.008"
      rotation="1 0 0 1.570796">
     <Shape>
      <Appearance USE="APP"/>
      <Cone bottomRadius="0.004" height="0.015"/>
     </Shape>
    </Transform>
   </Group>
  """ )[0]


class ClipPlanes(TypedField(AutoUpdate(SFBool),
                            (SFBool,SFVec3f,SFRotation))):
  """ClipPlanes allows the user to interactively control clipplanes
  in a scene. A button click adds a clipplane at the current
  position with the current orientation. The plane can then be moved
  and rotated. Previously added planes can be interactively moved
  and rotated, and removed.
  
  The following must be provided through the "references" field:
  1) the group node which should be clipped

  The following may also be provided through the "references" field:
  1) a identically transformed group for clipplane icons,
  2) an identically transformed LocalInfo node,
  3) an icon for the clipplanes.
  
  To the provided field clipPlanes the following must be routed:
  1) the button to control the clipplanes with,
  2) the position to control the clipplanes, and
  3) the orientation to control the clipplanes.
  """
    
  def __init__(self):
    AutoUpdate(SFBool).__init__(self)
    self.button = 0
    self.clipplanes = []
    
  def update(self,event):
    try:
      input = self.getRoutesIn()
      button = input[0].getValue()
      position = input[1].getValue()
      orientation = input[2].getValue()
      self_button = self.button
    except:
      return 0

    
    if button == 1 and self_button == 0:

      # Note on using this script multiple times with several volumes:
      # Bounding box checking with the volumes is needed to see in which
      # volume space we are when placing a new clipping plane.
      # For moving the planes we will encounter problems when two planes
      # From the different volumes are within grabing distance; both
      # planes will be moved. This might be resolved with a global switch that
      # tells us the name of the selected volume, provided we can access the volume
      # through here in order to know which copy of the script we are accessing
      
      self.current_plane = None
      for clipplane in self.clipplanes:
        if ( clipplane[0].point.getValue()
             - position ).length() < GRAB_DISTANCE * self.scaling:
          self.current_plane = clipplane
      
      if self.current_plane == None:
        plane = createX3DNodeFromString( """
         <PointNormalClipPlane />
        """ )[0]

        plane_shape = createX3DNodeFromString( """
         <Transform>
         </Transform>
        """ )[0]
        plane_shape.children.push_back(icon)
        
        if info != None:
          M = info.accInverseMatrix.getValue().getScaleRotationPart()
          s = 0.57735 * Vec3f( ( M * Vec3f(1,0,0) ).length(),
                               ( M * Vec3f(0,1,0) ).length(),
                               ( M * Vec3f(0,0,1) ).length() ).length()
          plane_shape.scale.setValue( Vec3f(s,s,s) )
          self.scaling = s
        
        children = groupA.children.getValue()
        children.insert( len(self.clipplanes), plane )
        groupA.children.setValue(children)
        
        children = groupB.children.getValue()
        children.insert( len(self.clipplanes), plane_shape )
        groupB.children.setValue(children)
        
        self.clipplanes.append( (plane,plane_shape) )
        self.current_plane = self.clipplanes[-1]

        self.newly_added = True
        
      else:
        self.newly_added = False
      
      self.current_plane[0].point.setValue( position )
      self.current_plane[0].normal.setValue( Matrix3f(orientation)
                                             * Vec3f(0,0,-1) )
      self.current_plane[1].translation.setValue( position )
      self.current_plane[1].rotation.setValue( orientation )
      
      self.start_position = position
      self.start_orientation = orientation

      transformChildren = self.current_plane[1].children.getValue()
      transformChildren[0].children.getValue()[0].springConstant.setValue(0)
      self.current_plane[1].children.setValue(transformChildren)

    elif button == 1 and self_button == 1:

      self.current_plane[0].point.setValue( position )
      self.current_plane[0].normal.setValue( Matrix3f(orientation)
                                             * Vec3f(0,0,-1) )
      self.current_plane[1].translation.setValue( position )
      self.current_plane[1].rotation.setValue( orientation )

      transformChildren = self.current_plane[1].children.getValue()
      transformChildren[0].children.getValue()[0].springConstant.setValue(0)
      self.current_plane[1].children.setValue(transformChildren)

    elif button == 0 and self_button == 1:

      transformChildren = self.current_plane[1].children.getValue()
      transformChildren[0].children.getValue()[0].springConstant.setValue(200)
      self.current_plane[1].children.setValue(transformChildren)

      if not self.newly_added \
         and ( position - self.start_position ).length() \
             < SMALL_DISTANCE * self.scaling \
         and ( Matrix3f(self.start_orientation) * Vec3f(1,0,0) -
               Matrix3f(orientation) * Vec3f(1,0,0) ).length() < 0.1:

        children = groupA.children.getValue()
        children.remove( self.current_plane[0] )
        groupA.children.setValue(children)
      
        children = groupB.children.getValue()
        children.remove( self.current_plane[1] )
        groupB.children.setValue(children)
      
        self.clipplanes.remove( self.current_plane )
        self.current_plane = None
    
    self.button = button
    return 1


clipPlanes = ClipPlanes()

