/***********************************************************************
* Project:      VolViewer
* Title:        VolumeRender.h
* Author:       Jerome Avondo
*
* Purpose:      Rendering class for a volume object, uses qt and openGL
*				Currently support for 2D texture maps and 3D texture maps
*
************************************************************************/

#ifndef VOLREN_H
#define VOLREN_H

#include <qgl.h>
#include <qimage.h> 
#include <qstring.h> 
#include <qmessagebox.h>
#include <qtimer.h>
#include <qdatetime.h>
#include <qcolordialog.h>
#include <QThreadPool>

#include <math.h>
#include <vector>
using namespace std;

#include "glext.h"

#include "VolumeObject.h"
#include "VolIO.h"
#include "Vector.h"
#include "Camera.h"
#include "ArcBall.h"
#include "MeshListObject.h"
#include "Matrix4x4.h"
#include "PlaneWidgetInCube.h"
#include "PlaneWidget.h" 
//#include "LineSegmentObject.h"
//#include "FTGL_FontManager.h"
#include "ColourFun.h"
#include "GL_Ext.h"
#include "GL_ShaderObj.h"
#include "GL_FrameBufferObject.h"
#include "Timer.h"
#include "PCA.h"
#include "Triangulate.h"
#include "CVector.h"
#include "Edge.h"

#include "QtTransferFunctionEditorDialogue.h"
#include "QtThreadedBrickLoader.h"


enum {X, Y, Z, W};

class VolumeRender
{

public:

	GLuint SLICEX;
	GLuint SLICEY;
	GLuint SLICEZ;

	GLuint TEXTURE3D_RED;
	GLuint TEXTURE3D_GREEN;
	GLuint TEXTURE3D_BLUE;
	GLuint TEXTURE2D_RGB;
	GLuint TEXTURE2D_RAGABA;

	GLuint TEXTURE2D_FBO_FB;
	GLuint TEXTURE2D_FBO_DP;
	GLuint TEXTURE2D_LODFBO_FB;
	GLuint TEXTURE2D_LODFBO_DP;
	GLuint TEXTURE2D_JITTER;

	GLuint TEXTURE3D_PBO;

	unsigned int* depthdata;

	Vector v1, v2, v3, v4;
	Vector t1, t2, t3, t4;
//	float maxres;
	int numbslices;
	
	QTime t;
	Timer timer;

	int benchmark;
	int testerJA;

	int is_greyscale;
	int r_channel, g_channel, b_channel, isosurface;

	int FBO_SIZE;
	int LOD_FBO_SIZE;

	int render_size;
	int fbo_lod;
	int fbo_depth;

	Vector goochwarm;
	Vector goochcool;
	float goochcontrib;
	float goochexp;
	float goochalpha;
	float goochdegen;

	//Vector intersect;	
	vector<Vector> celleigenvectors;
	vector<Vector> cellmeans;
	
	vector< vector<Vector> > time_celleigenvectors;
	vector< vector<Vector> > time_cellmeans;
	vector< vector< vector<bool> > > time_celledges;

	//frame buffer object,
	//for render scene to texture
	int vbo_size;
	GL_FrameBufferObject fbo_object;
	GL_FrameBufferObject lodfbo_object;

	//rebuild texcoords,
	Vector texrot_t1, texrot_t2, texrot_t3, texrot_t4;
	Matrix4x4 texrot_mat;
	float texrot_s;
	int texrot_index;
	int texrot_i;
	int texrot_indexhelper;

	int updatehisto;

	//bright & cont for tfuncts
	int l_bright;
	int l_cont;
	int la_bright;
	int la_cont;
	int r_bright;
	int r_cont;
	int g_bright;
	int g_cont;
	int b_bright;
	int b_cont;
	int ra_bright;
	int ra_cont;
	int ga_bright;
	int ga_cont;
	int ba_bright;
	int ba_cont;

	//threholds for tfuncts;
	int l_tmax;
	int l_tmin;
	int la_tmax;
	int la_tmin;
	int r_tmax;
	int r_tmin;
	int g_tmax;
	int g_tmin;
	int b_tmax;
	int b_tmin;
	int ra_tmax;
	int ra_tmin;
	int ga_tmax;
	int ga_tmin;
	int ba_tmax;
	int ba_tmin;

	bool inv_l;
	bool inv_la;
	bool inv_r;
	bool inv_g;
	bool inv_b;
	bool inv_ra;
	bool inv_ga;
	bool inv_ba;

	int plane_rendermode[6];

	//transfer functions
	unsigned char rgb_tfunc[256*3];
	unsigned char ragaba_tfunc[256*3];

	//a backup copy of transfer functions
	//used to make changing threshold values quicker
	//so we have to send less data between widgets.
	unsigned char backup_rgb_tfunc[256*3];
	unsigned char backup_ragaba_tfunc[256*3];

	//background colour
	QColor old_background_colour;
	QColor background_colour;
	QColor background_fontcolour;

	//bounding box colour
	QColor bb_colour;

	float angl;

	//brick rendering stuff (EXPERIMENTAL)
	int loaded;
	int brick;
	int BRICKS_SUBDIV;
	//vector < PlaneWidgetInCube > brick_cutting_planes;
	//vector < vector<bool> > brick_cutting_planes_clipstate;

	//PlaneWidgetInCube brick_cutting_planes;
	//vector<GLuint> brick_texids;
	//vector<GLuint> brick_pbotexids;
	
	vector< Vector > brick_centresrot;
	vector< int > sorted_brick_indices;
	vector< bool > empty_brick;
	vector< Vector > brick_centres;
	vector< vector< unsigned char > > brick_textures;

	//pointer to our remapdialogue
	QtTransferFunctionEditorDialogue* editor;

	//Frame rate
	QTime fpsTime_;
	unsigned int fpsCounter_;
	QString fpsString_;
	float f_p_s_;

	//real world units to voxel resolution
	float res;
	
	//gl extensions
	GL_Ext			glextensions;

	//volume rendering shader, isosurface shader, 
	//post process scene image shader
	int				volshader_id;
	GL_ShaderOBJ	vol_sectionshaderobj;
	GL_ShaderOBJ	vol_shaderobj;
	GL_ShaderOBJ	vol2d_shaderobj;
	GL_ShaderOBJ	vol2dmulti_shaderobj;
	GL_ShaderOBJ	vol_rgbsectionshaderobj;
	GL_ShaderOBJ	vol_shadersinglepassobj;
	GL_ShaderOBJ	mesh_shaderobj;
	GL_ShaderOBJ	postrendering_shaderobj;

	//camera 
	ArcBall arcball;

	//cropping
	bool crop_inv;
	int crop_shape_selected;
	int cropping_mode;
	vector<Vector> crop_triangles;
	vector<Vector> crop_shape2d;
	vector<Vector> crop_shape;
	vector<Vector> crop_normals;
	vector<Vector> crop_centroids;

	int measure_vertlabel;
	int measure_lengthlabel;
	int measure_anglelabel;

	//measuring objects
	int measure_editmode;
	int measure_dragmode;
	int measure_modify;
	int measure_mode;

	Vector current_measure_centre;
	int current_measure_drawmode;
	int current_measure_index;
	int current_measure_vertindex;
	int measure_magnetline_vert_old;
	int measure_magnetline_vert;
	int measure_magnetline_edge;
	int measure_magnetline_destvert;
	int measure_magnetline_helper;

	vector<Vector> current_measure_points;
	vector<Vector> current_normals;
	vector<Edge> current_edges;

	vector<int> measure_objects_visible;
	vector<bool> measure_objects_drawmode;
	vector< vector<Vector> > measure_objects;
	vector< vector<Vector> > measure_normals;
	vector< vector<Edge> > measure_edges;
	
	int measure_region_size_min;
	int measure_region_size_max;
	vector<int> measure_region_sizes;
	vector<Vector> measure_region_means;

	Vector scalebarfontpos;
	QString scalebartext;

	//used for mouse click unproject in qtglwidget
	double mvmatrix[16];
	double projmatrix[16];
	int viewport[4];

	//string containing our screenspace cursor coord;
	QString cursor3D; 
	
	//3D Cursor position in screen space
	Vector cursor3D_screenspace;
	
	//3D Cursor position in texture space
	Vector cursor3D_texturespace;
	
	//modelview matrix
	float modelMatrix[16];
	
	//pointer to our volume object
	VolumeObject* volobj;
	VolIO* volio;

	//list of mesh objects
	MeshListObject meshlist;

	// window dimensions
	int winWidth, winHeight, winMaxDim;

	//our bounding box (8 vertices)
	Vector bb_v1, bb_v2, bb_v3, bb_v4, bb_v5, bb_v6, bb_v7, bb_v8;

	// 2d texture variables
	GLuint *texNames1, *texNames2, *texNames3;
	float maxX, maxY, maxZ, minX, minY, minZ;
	int fillSlices;
    GLfloat a1,a2,a3;

	// 3d texture variables
	int slices, sliceslod, max_slices;
    GLfloat offS, offT, offR; 	
	GLfloat splane[4], rplane[4], tplane[4];
	
	//slice blending mode
	int bldmode;

	//viewing parameters
	Camera camera;
	float zoom;
	float tx;
	float ty;	
	float tz;	
	int x_slice, y_slice, z_slice;

	//tx, ty in texture space
	Vector tex_translate;

	//Cutting Planes Widget
	PlaneWidgetInCube cutting_planes;

	//various rendering toggles
	int	lights_toggle, volume_toggle, stereo_mode, stereo_toggle, volume_render_mode, isosurface_toggle, BB_toggle;
	int scalebar_toggle, measure_toggle, cursor_toggle, fps_toggle, axis_toggle, redclip_toggle, blueclip_toggle, greenclip_toggle, isoclip_toggle, projection_toggle, FBOmoveLOD_toggle, SlicesmoveLOD_toggle, PostProcess_toggle;
	
	int offscreen_toggle;

	float stereo_eye_seperation;

	GLenum slice_draw_mode;

	int timelapsefolder;
	int timelapsefoldersize;

	//lighting
	int lighting_enable;
	Vector light_pos0;
	Vector light_dir0;
	Vector light_pos1;
	Vector light_dir1;
	float light_radius, light_azimuth, light_altitude;

	//silhouttes
	int silhouette_enable;
	float silhouette_amount;

	//gooch tone lighting
	int gooch_enable;
	float gooch_toneCont;
	float gooch_warmCol[3];
	float gooch_coolCol[3];

	//2d post process
	float postprocess2d_amount;

	VolumeRender();
	~VolumeRender();

	//----------------------------------------------------
	//Main OpenGL functions
	//----------------------------------------------------
	void		initializeGL();
    void		displayGL(int render);
    void		resizeGL( int w, int h );

	void		initvariables();
	void		init_FBO(void);

	void		rebuild_texcoords(void);
	void		rebuildwithscaling_texcoords(void);

	//----------------------------------------------------
	//Texture Loading
	//----------------------------------------------------
	void		load_3dTexture_brick(int brickindex);
	void		load_TextureMap(void);
	void		load_2DTextureMaps(void);
	void		load_3DTextureMap(void);
	void		load_1DLookupTables(void);
	void		load_1DLookupTablesBricks(int brickres, int brickslices);
	void		load_2DJitterTables(void);

	void		update_FBOsize(int size);
	void		select_channel(int channel);

	//----------------------------------------------------
	//Volume Rendering
	//----------------------------------------------------
	//Brick rendering
	void		sort_bricks(void);
	void		init_3dTexture_bricks(int subdivs);
	void		render_3dTexture_bricks(int subdivs);
	//3D Texture Single Pass (RGB)
	void		render_3dTexture(void);
	//2D Texture Single Pass (RGB)
	void		render_2dTexture(void);
	void		render_2dTexture_stack1(GLfloat dir);
	void		render_2dTexture_stack2(GLfloat dir);
	void		render_2dTexture_stack3(GLfloat dir);
	//2D Multi-Texture MultiPass
	void		render_2dMultiTextureMultiPass(void);
	void		render_2dMultiTextureMultiPass_stack1(GLfloat dir);
	void		render_2dMultiTextureMultiPass_stack2(GLfloat dir);
	void		render_2dMultiTextureMultiPass_stack3(GLfloat dir);
	//3D Texture MultiPass
	void		render_3dTextureMultiPass(void);

	//misc
	void		render_celleigenvectors(void);

	//clipping plane slices
	void		render_3dTextureSlice1(void);
	void		render_3dTextureSlice(void);
	void		render_3dTextureSlicePlane_i(int i, GLenum surface, int mode, int text);
	void		render_3dTextureSlicePlane_i_bricks(int i, GLenum surface, int mode, int text);

	QImage		render_framebuffer(bool withAlpha);
	void		render_meshes(void);
	void		render_slices(GLenum draw_mode);
	void		render_boundingbox(void);
	void		render_scalebar(void);
	void		render_3DCursor(void);
	void        render_stereo(int render);
	void		render_scene(int render);
	void		render_FPS(void);
	void		render_axis(void);
	void		render_light(Vector lightpos, Vector lightdir);
	void		render_measurements(void);
	void		render_aabb(Vector cen, Vector dim);
	void		render_sphere(Vector c,float r,int n, GLenum drawmode);
	void		render_ellipsoid(Vector c,Vector r, Matrix4x4* rotmat, int n, GLenum drawmode);

	void		blend_mode(int blendmode);
	void		check_extensions(void);


	//----------------------------------------------------
	//Transfer Functions
	//----------------------------------------------------
	void		reset_transfer_function_parameters(int _r_tmax, int _r_tmin, int _g_tmax, int _g_tmin, int _b_tmax, int _b_tmin, int _ra_tmax, int _ra_tmin, int _ga_tmax, int _ga_tmin, int _ba_tmax, int _ba_tmin,
				    							   int _r_bright, int _r_cont, int _g_bright, int _g_cont, int _b_bright, int _b_cont, int _ra_bright, int _ra_cont, int _ga_bright, int _ga_cont, int _ba_bright, int _ba_cont,
												   bool _inv_r,	bool _inv_g, bool _inv_b, bool _inv_ra, bool _inv_ga, bool _inv_ba);
	void		apply_transfer_function(const vector<Vector>& tfunc_curvepts, float graphsize, float border, int mode);
	void		apply_transfer_function_all(const vector<Vector>& r_tfunc_curvepts,  const vector<Vector>& g_tfunc_curvepts,  const vector<Vector>& b_tfunc_curvepts, 
											const vector<Vector>& ra_tfunc_curvepts, const vector<Vector>& ga_tfunc_curvepts, const vector<Vector>& ba_tfunc_curvepts,
											float graphsize, float border);
	void		apply_invert(int mode, int togglestate);
	void		apply_invert16(int mode, int togglestate);
	void		apply_threshold(int mode, int tmin, int tmax);
	void		apply_brightcontrast(int mode, int b, int c);	
	void		build_histogram(void);

	//----------------------------------------------------
	//Camera
	//----------------------------------------------------
	void		setViewfromEuler(float angleX, float angleY, float angleZ);
	void		setViewfromEuler2(float angleX, float angleY, float angleZ);

	//----------------------------------------------------
	//Shaders
	//----------------------------------------------------
	void		aim_light(void);
	void		light_colours(Vector amb, Vector diff, Vector spec);
	void		silhouettes_shader(void);
	void		postprocess2d_shader(void);
	void		silhouette_shader(void);
	void		volshader_event(int id);
	void		shaderGoochEvent(Vector warm, Vector cool, int contrib, int expon, int alpha, int degen);
};

#endif //VOLREN_H
