/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.tiff;

import java.io.IOException;
import java.util.TreeSet;
import java.util.Vector;
import loci.common.DataTools;
import loci.common.RandomAccessInputStream;
import loci.common.Region;
import loci.common.enumeration.EnumException;
import loci.formats.FormatException;
import loci.formats.codec.BitBuffer;
import loci.formats.codec.CodecOptions;
import loci.formats.tiff.IFD;
import loci.formats.tiff.IFDList;
import loci.formats.tiff.IFDType;
import loci.formats.tiff.PhotoInterp;
import loci.formats.tiff.TiffCompression;
import loci.formats.tiff.TiffIFDEntry;
import loci.formats.tiff.TiffRational;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TiffParser {
    private static final Logger LOGGER = LoggerFactory.getLogger(TiffParser.class);
    protected RandomAccessInputStream in;
    private byte[] cachedTileBuffer;
    private boolean bigTiff;
    private boolean doCaching;
    private IFDList ifdList;
    private IFD firstIFD;

    public TiffParser(String filename) throws IOException {
        this(new RandomAccessInputStream(filename));
    }

    public TiffParser(RandomAccessInputStream in) {
        this.in = in;
        this.doCaching = true;
        try {
            long fp = in.getFilePointer();
            this.checkHeader();
            in.seek(fp);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public void setDoCaching(boolean doCaching) {
        this.doCaching = doCaching;
    }

    public RandomAccessInputStream getStream() {
        return this.in;
    }

    public boolean isValidHeader() {
        try {
            return this.checkHeader() != null;
        }
        catch (IOException e) {
            return false;
        }
    }

    public Boolean checkHeader() throws IOException {
        boolean bigEndian;
        if (this.in.length() < 4L) {
            return null;
        }
        this.in.seek(0L);
        int endianOne = this.in.read();
        int endianTwo = this.in.read();
        boolean littleEndian = endianOne == 73 && endianTwo == 73;
        boolean bl = bigEndian = endianOne == 77 && endianTwo == 77;
        if (!littleEndian && !bigEndian) {
            return null;
        }
        this.in.order(littleEndian);
        short magic = this.in.readShort();
        boolean bl2 = this.bigTiff = magic == 43;
        if (magic != 42 && magic != 43) {
            return null;
        }
        return new Boolean(littleEndian);
    }

    public boolean isBigTiff() {
        return this.bigTiff;
    }

    public IFDList getIFDs() throws IOException {
        if (this.ifdList != null) {
            return this.ifdList;
        }
        long[] offsets = this.getIFDOffsets();
        IFDList ifds = new IFDList();
        for (long offset : offsets) {
            IFD ifd = this.getIFD(offset);
            if (ifd == null) continue;
            if (ifd.containsKey(256)) {
                ifds.add(ifd);
            }
            long[] subOffsets = null;
            try {
                subOffsets = ifd.getIFDLongArray(330);
            }
            catch (FormatException e) {
                // empty catch block
            }
            if (subOffsets == null) continue;
            for (long subOffset : subOffsets) {
                IFD sub = this.getIFD(subOffset);
                if (sub == null) continue;
                ifds.add(sub);
            }
        }
        if (this.doCaching) {
            this.ifdList = ifds;
        }
        return ifds;
    }

    public IFDList getThumbnailIFDs() throws IOException {
        IFDList ifds = this.getIFDs();
        IFDList thumbnails = new IFDList();
        for (IFD ifd : ifds) {
            Number subfile = (Number)ifd.getIFDValue(254);
            int subfileType = subfile == null ? 0 : subfile.intValue();
            if (subfileType != 1) continue;
            thumbnails.add(ifd);
        }
        return thumbnails;
    }

    public IFDList getNonThumbnailIFDs() throws IOException {
        IFDList ifds = this.getIFDs();
        IFDList nonThumbs = new IFDList();
        for (IFD ifd : ifds) {
            int subfileType;
            Number subfile = (Number)ifd.getIFDValue(254);
            int n = subfileType = subfile == null ? 0 : subfile.intValue();
            if (subfileType == 1 && ifds.size() > 1) continue;
            nonThumbs.add(ifd);
        }
        return nonThumbs;
    }

    public IFDList getExifIFDs() throws FormatException, IOException {
        IFDList ifds = this.getIFDs();
        IFDList exif = new IFDList();
        for (IFD ifd : ifds) {
            IFD exifIFD;
            long offset = ifd.getIFDLongValue(34665, 0L);
            if (offset == 0L || (exifIFD = this.getIFD(offset)) == null) continue;
            exif.add(exifIFD);
        }
        return exif;
    }

    public long[] getIFDOffsets() throws IOException {
        int bytesPerEntry = this.bigTiff ? 20 : 12;
        Vector<Long> offsets = new Vector<Long>();
        long offset = this.getFirstOffset();
        while (offset > 0L && offset < this.in.length()) {
            this.in.seek(offset);
            offsets.add(offset);
            int nEntries = this.bigTiff ? (int)this.in.readLong() : this.in.readUnsignedShort();
            this.in.skipBytes(nEntries * bytesPerEntry);
            offset = this.getNextOffset(offset);
        }
        long[] f = new long[offsets.size()];
        for (int i = 0; i < f.length; ++i) {
            f[i] = (Long)offsets.get(i);
        }
        return f;
    }

    public IFD getFirstIFD() throws IOException {
        if (this.firstIFD != null) {
            return this.firstIFD;
        }
        long offset = this.getFirstOffset();
        IFD ifd = this.getIFD(offset);
        if (this.doCaching) {
            this.firstIFD = ifd;
        }
        return ifd;
    }

    public TiffIFDEntry getFirstIFDEntry(int tag) throws IOException {
        long offset = this.getFirstOffset();
        if (offset < 0L) {
            return null;
        }
        this.in.seek(offset);
        long numEntries = this.bigTiff ? this.in.readLong() : (long)this.in.readUnsignedShort();
        int i = 0;
        while ((long)i < numEntries) {
            this.in.seek(offset + (long)(this.bigTiff ? 8 : 2) + (long)((this.bigTiff ? 20 : 12) * i));
            TiffIFDEntry entry = this.readTiffIFDEntry();
            if (entry.getTag() == tag) {
                return entry;
            }
            ++i;
        }
        throw new IllegalArgumentException("Unknown tag: " + tag);
    }

    public long getFirstOffset() throws IOException {
        Boolean header = this.checkHeader();
        if (header == null) {
            return -1L;
        }
        if (this.bigTiff) {
            this.in.skipBytes(4);
        }
        return this.getNextOffset(0L);
    }

    public IFD getIFD(long offset) throws IOException {
        if (offset < 0L || offset >= this.in.length()) {
            return null;
        }
        IFD ifd = new IFD();
        ifd.put(new Integer(0), new Boolean(this.in.isLittleEndian()));
        ifd.put(new Integer(1), new Boolean(this.bigTiff));
        LOGGER.trace("getIFDs: seeking IFD at {}", (Object)offset);
        this.in.seek(offset);
        long numEntries = this.bigTiff ? this.in.readLong() : (long)this.in.readUnsignedShort();
        LOGGER.trace("getIFDs: {} directory entries to read", (Object)numEntries);
        if (numEntries == 0L || numEntries == 1L) {
            return ifd;
        }
        int bytesPerEntry = this.bigTiff ? 20 : 12;
        int baseOffset = this.bigTiff ? 8 : 2;
        int i = 0;
        while ((long)i < numEntries) {
            this.in.seek(offset + (long)baseOffset + (long)(bytesPerEntry * i));
            TiffIFDEntry entry = null;
            try {
                entry = this.readTiffIFDEntry();
            }
            catch (EnumException e) {
                LOGGER.debug("", (Throwable)e);
            }
            if (entry == null) break;
            int count = entry.getValueCount();
            int tag = entry.getTag();
            long pointer = entry.getValueOffset();
            int bpe = entry.getType().getBytesPerElement();
            if (count < 0 || bpe <= 0) {
                this.in.skipBytes(bytesPerEntry - 4 - (this.bigTiff ? 8 : 4));
            } else {
                Object value = null;
                long inputLen = this.in.length();
                if ((long)(count * bpe) + pointer > inputLen) {
                    int oldCount = count;
                    count = (int)((inputLen - pointer) / (long)bpe);
                    LOGGER.trace("getIFDs: truncated {} array elements for tag {}", (Object)(oldCount - count), (Object)tag);
                    if (count < 0) {
                        count = oldCount;
                    }
                }
                if (count < 0 || (long)count > this.in.length()) break;
                value = pointer != this.in.getFilePointer() && !this.doCaching ? entry : this.getIFDValue(entry);
                if (value != null && !ifd.containsKey(new Integer(tag))) {
                    ifd.put(new Integer(tag), value);
                }
            }
            ++i;
        }
        this.in.seek(offset + (long)baseOffset + (long)bytesPerEntry * numEntries);
        return ifd;
    }

    public void fillInIFD(IFD ifd) throws IOException {
        TreeSet<TiffIFDEntry> entries = new TreeSet<TiffIFDEntry>();
        for (Integer key : ifd.keySet()) {
            if (!(ifd.get(key) instanceof TiffIFDEntry)) continue;
            entries.add((TiffIFDEntry)ifd.get(key));
        }
        for (TiffIFDEntry entry : entries) {
            ifd.put(new Integer(entry.getTag()), this.getIFDValue(entry));
        }
    }

    public Object getIFDValue(TiffIFDEntry entry) throws IOException {
        IFDType type = entry.getType();
        int count = entry.getValueCount();
        long offset = entry.getValueOffset();
        LOGGER.trace("Reading entry {} from {}; type={}, count={}", new Object[]{entry.getTag(), offset, type, count});
        if (offset >= this.in.length()) {
            return null;
        }
        if (offset != this.in.getFilePointer()) {
            this.in.seek(offset);
        }
        if (type == IFDType.BYTE) {
            if (count == 1) {
                return new Short(this.in.readByte());
            }
            byte[] bytes = new byte[count];
            this.in.readFully(bytes);
            short[] shorts = new short[count];
            for (int j = 0; j < count; ++j) {
                shorts[j] = (short)(bytes[j] & 0xFF);
            }
            return shorts;
        }
        if (type == IFDType.ASCII) {
            byte[] ascii = new byte[count];
            this.in.read(ascii);
            int nullCount = 0;
            for (int j = 0; j < count; ++j) {
                if (ascii[j] != 0 && j != count - 1) continue;
                ++nullCount;
            }
            String[] strings = nullCount == 1 ? null : new String[nullCount];
            Object s = null;
            int c = 0;
            int ndx = -1;
            for (int j = 0; j < count; ++j) {
                if (ascii[j] == 0) {
                    s = new String(ascii, ndx + 1, j - ndx - 1);
                    ndx = j;
                } else {
                    s = j == count - 1 ? new String(ascii, ndx + 1, j - ndx) : null;
                }
                if (strings == null || s == null) continue;
                strings[c++] = s;
            }
            return strings == null ? s : strings;
        }
        if (type == IFDType.SHORT) {
            if (count == 1) {
                return new Integer(this.in.readUnsignedShort());
            }
            int[] shorts = new int[count];
            for (int j = 0; j < count; ++j) {
                shorts[j] = this.in.readUnsignedShort();
            }
            return shorts;
        }
        if (type == IFDType.LONG || type == IFDType.IFD) {
            if (count == 1) {
                return new Long(this.in.readInt());
            }
            long[] longs = new long[count];
            for (int j = 0; j < count; ++j) {
                if (this.in.getFilePointer() + 4L > this.in.length()) continue;
                longs[j] = this.in.readInt();
            }
            return longs;
        }
        if (type == IFDType.LONG8 || type == IFDType.SLONG8 || type == IFDType.IFD8) {
            if (count == 1) {
                return new Long(this.in.readLong());
            }
            long[] longs = new long[count];
            for (int j = 0; j < count; ++j) {
                longs[j] = this.in.readLong();
            }
            return longs;
        }
        if (type == IFDType.RATIONAL || type == IFDType.SRATIONAL) {
            if (count == 1) {
                return new TiffRational(this.in.readInt(), this.in.readInt());
            }
            TiffRational[] rationals = new TiffRational[count];
            for (int j = 0; j < count; ++j) {
                rationals[j] = new TiffRational(this.in.readInt(), this.in.readInt());
            }
            return rationals;
        }
        if (type == IFDType.SBYTE || type == IFDType.UNDEFINED) {
            if (count == 1) {
                return new Byte(this.in.readByte());
            }
            byte[] sbytes = new byte[count];
            this.in.read(sbytes);
            return sbytes;
        }
        if (type == IFDType.SSHORT) {
            if (count == 1) {
                return new Short(this.in.readShort());
            }
            short[] sshorts = new short[count];
            for (int j = 0; j < count; ++j) {
                sshorts[j] = this.in.readShort();
            }
            return sshorts;
        }
        if (type == IFDType.SLONG) {
            if (count == 1) {
                return new Integer(this.in.readInt());
            }
            int[] slongs = new int[count];
            for (int j = 0; j < count; ++j) {
                slongs[j] = this.in.readInt();
            }
            return slongs;
        }
        if (type == IFDType.FLOAT) {
            if (count == 1) {
                return new Float(this.in.readFloat());
            }
            float[] floats = new float[count];
            for (int j = 0; j < count; ++j) {
                floats[j] = this.in.readFloat();
            }
            return floats;
        }
        if (type == IFDType.DOUBLE) {
            if (count == 1) {
                return new Double(this.in.readDouble());
            }
            double[] doubles = new double[count];
            for (int j = 0; j < count; ++j) {
                doubles[j] = this.in.readDouble();
            }
            return doubles;
        }
        return null;
    }

    public String getComment() throws IOException {
        IFD firstIFD = this.getFirstIFD();
        return firstIFD == null ? null : firstIFD.getComment();
    }

    public byte[] getTile(IFD ifd, byte[] buf, int row, int col) throws FormatException, IOException {
        long[] rowsPerStrip;
        int tileNumber;
        byte[] jpegTable = (byte[])ifd.getIFDValue(347);
        CodecOptions options = new CodecOptions();
        options.interleaved = true;
        options.littleEndian = ifd.isLittleEndian();
        long tileWidth = ifd.getTileWidth();
        long tileLength = ifd.getTileLength();
        int samplesPerPixel = ifd.getSamplesPerPixel();
        int planarConfig = ifd.getPlanarConfiguration();
        TiffCompression compression = ifd.getCompression();
        long numTileCols = ifd.getTilesPerRow();
        int pixel = ifd.getBytesPerSample()[0];
        int effectiveChannels = planarConfig == 2 ? 1 : samplesPerPixel;
        long[] stripOffsets = ifd.getStripOffsets();
        long[] stripByteCounts = ifd.getStripByteCounts();
        if (stripByteCounts[tileNumber = (int)((long)row * numTileCols + (long)col)] == (rowsPerStrip = ifd.getRowsPerStrip())[0] * tileWidth && pixel > 1) {
            int n = tileNumber;
            stripByteCounts[n] = stripByteCounts[n] * (long)pixel;
        }
        int size = (int)(tileWidth * tileLength * (long)pixel * (long)effectiveChannels);
        if (buf == null) {
            buf = new byte[size];
        }
        if (stripByteCounts[tileNumber] == 0L || stripOffsets[tileNumber] >= this.in.length()) {
            return buf;
        }
        byte[] tile = new byte[(int)stripByteCounts[tileNumber]];
        this.in.seek(stripOffsets[tileNumber]);
        this.in.read(tile);
        options.maxBytes = Math.max(size, tile.length);
        if (jpegTable != null) {
            byte[] q = new byte[jpegTable.length + tile.length - 4];
            System.arraycopy(jpegTable, 0, q, 0, jpegTable.length - 2);
            System.arraycopy(tile, 2, q, jpegTable.length - 2, tile.length - 2);
            tile = compression.decompress(q, options);
        } else {
            tile = compression.decompress(tile, options);
        }
        TiffCompression.undifference(tile, ifd);
        TiffParser.unpackBytes(buf, 0, tile, ifd);
        return buf;
    }

    public byte[] getSamples(IFD ifd, byte[] buf) throws FormatException, IOException {
        long width = ifd.getImageWidth();
        long length = ifd.getImageLength();
        return this.getSamples(ifd, buf, 0, 0, width, length);
    }

    public byte[] getSamples(IFD ifd, byte[] buf, int x, int y, long width, long height) throws FormatException, IOException {
        LOGGER.trace("parsing IFD entries");
        boolean littleEndian = ifd.isLittleEndian();
        this.in.order(littleEndian);
        int samplesPerPixel = ifd.getSamplesPerPixel();
        long tileWidth = ifd.getTileWidth();
        long tileLength = ifd.getTileLength();
        if (tileLength <= 0L) {
            LOGGER.trace("Tile length is {}; setting it to {}", (Object)tileLength, (Object)height);
            tileLength = height;
        }
        long numTileRows = ifd.getTilesPerColumn();
        long numTileCols = ifd.getTilesPerRow();
        PhotoInterp photoInterp = ifd.getPhotometricInterpretation();
        int planarConfig = ifd.getPlanarConfiguration();
        int pixel = ifd.getBytesPerSample()[0];
        int effectiveChannels = planarConfig == 2 ? 1 : samplesPerPixel;
        ifd.printIFD();
        if (width * height > Integer.MAX_VALUE) {
            throw new FormatException("Sorry, ImageWidth x ImageLength > 2147483647 is not supported (" + width + " x " + height + ")");
        }
        if (width * height * (long)effectiveChannels * (long)pixel > Integer.MAX_VALUE) {
            throw new FormatException("Sorry, ImageWidth x ImageLength x SamplesPerPixel x BitsPerSample > 2147483647 is not supported (" + width + " x " + height + " x " + samplesPerPixel + " x " + pixel * 8 + ")");
        }
        int numSamples = (int)(width * height);
        LOGGER.trace("reading image data (samplesPerPixel={}; numSamples={})", (Object)samplesPerPixel, (Object)numSamples);
        TiffCompression compression = ifd.getCompression();
        if ((long)x % tileWidth == 0L && (long)y % tileLength == 0L && width == tileWidth && height == tileLength && samplesPerPixel == 1 && ifd.getBitsPerSample()[0] % 8 == 0 && photoInterp != PhotoInterp.WHITE_IS_ZERO && photoInterp != PhotoInterp.CMYK && photoInterp != PhotoInterp.Y_CB_CR && compression == TiffCompression.UNCOMPRESSED) {
            long[] stripOffsets = ifd.getStripOffsets();
            long[] stripByteCounts = ifd.getStripByteCounts();
            if (stripOffsets != null && stripByteCounts != null) {
                int tile = (int)((long)y / tileLength * numTileCols + (long)x / tileWidth);
                if (stripByteCounts[tile] == (long)numSamples && pixel > 1) {
                    int n = tile;
                    stripByteCounts[n] = stripByteCounts[n] * (long)pixel;
                }
                this.in.seek(stripOffsets[tile]);
                this.in.read(buf, 0, (int)Math.min((long)buf.length, stripByteCounts[tile]));
            }
            return buf;
        }
        long nrows = numTileRows;
        if (planarConfig == 2) {
            numTileRows *= (long)samplesPerPixel;
        }
        Region imageBounds = new Region(x, y, (int)width, (int)(height * (long)(samplesPerPixel / effectiveChannels)));
        int endX = (int)width + x;
        int endY = (int)height + y;
        int rowLen = pixel * (int)tileWidth;
        int tileSize = (int)((long)rowLen * tileLength);
        int planeSize = (int)(width * height * (long)pixel);
        int outputRowLen = (int)((long)pixel * width);
        int bufferSizeSamplesPerPixel = samplesPerPixel;
        if (ifd.getPlanarConfiguration() == 2) {
            bufferSizeSamplesPerPixel = 1;
        }
        int bpp = ifd.getBytesPerSample()[0];
        int bufferSize = (int)tileWidth * (int)tileLength * bufferSizeSamplesPerPixel * bpp;
        if (this.cachedTileBuffer == null || this.cachedTileBuffer.length != bufferSize) {
            this.cachedTileBuffer = new byte[bufferSize];
        }
        Region tileBounds = new Region(0, 0, (int)tileWidth, (int)tileLength);
        int row = 0;
        while ((long)row < numTileRows) {
            int col = 0;
            while ((long)col < numTileCols) {
                tileBounds.x = col * (int)tileWidth;
                tileBounds.y = row * (int)tileLength;
                if (imageBounds.intersects(tileBounds)) {
                    if (planarConfig == 2) {
                        tileBounds.y = (int)((long)row % nrows * tileLength);
                    }
                    this.getTile(ifd, this.cachedTileBuffer, row, col);
                    int tileX = Math.max(tileBounds.x, x);
                    int tileY = Math.max(tileBounds.y, y);
                    int realX = tileX % (int)tileWidth;
                    int realY = tileY % (int)tileLength;
                    int twidth = (int)Math.min((long)(endX - tileX), tileWidth - (long)realX);
                    int theight = (int)Math.min((long)(endY - tileY), tileLength - (long)realY);
                    int copy = pixel * twidth;
                    realX *= pixel;
                    realY *= rowLen;
                    for (int q = 0; q < effectiveChannels; ++q) {
                        int src = q * tileSize + realX + realY;
                        int dest = q * planeSize + pixel * (tileX - x) + outputRowLen * (tileY - y);
                        if (planarConfig == 2) {
                            dest = (int)((long)dest + (long)planeSize * ((long)row / nrows));
                        }
                        for (int tileRow = 0; tileRow < theight; ++tileRow) {
                            System.arraycopy(this.cachedTileBuffer, src, buf, dest, copy);
                            src += rowLen;
                            dest += outputRowLen;
                        }
                    }
                }
                ++col;
            }
            ++row;
        }
        return buf;
    }

    public static void unpackBytes(byte[] samples, int startIndex, byte[] bytes, IFD ifd) throws FormatException {
        int ndx;
        int skipBits;
        boolean planar = ifd.getPlanarConfiguration() == 2;
        TiffCompression compression = ifd.getCompression();
        PhotoInterp photoInterp = ifd.getPhotometricInterpretation();
        if (compression == TiffCompression.JPEG) {
            photoInterp = PhotoInterp.RGB;
        }
        int[] bitsPerSample = ifd.getBitsPerSample();
        int nChannels = bitsPerSample.length;
        int sampleCount = 8 * bytes.length / (nChannels * bitsPerSample[0]);
        if (photoInterp == PhotoInterp.Y_CB_CR) {
            sampleCount *= 3;
        }
        if (planar) {
            sampleCount *= nChannels;
            nChannels = 1;
        }
        LOGGER.trace("unpacking {} samples (startIndex={}; totalBits={}; numBytes={})", new Object[]{sampleCount, startIndex, nChannels * bitsPerSample[0], bytes.length});
        long imageWidth = ifd.getImageWidth();
        long imageHeight = ifd.getImageLength();
        int bps0 = bitsPerSample[0];
        int numBytes = ifd.getBytesPerSample()[0];
        int nSamples = samples.length / (nChannels * numBytes);
        boolean noDiv8 = bps0 % 8 != 0;
        boolean bps8 = bps0 == 8;
        boolean bps16 = bps0 == 16;
        boolean littleEndian = ifd.isLittleEndian();
        BitBuffer bb = new BitBuffer(bytes);
        if ((bps8 || bps16) && bytes.length <= samples.length && nChannels == 1 && photoInterp != PhotoInterp.WHITE_IS_ZERO && photoInterp != PhotoInterp.CMYK && photoInterp != PhotoInterp.Y_CB_CR) {
            System.arraycopy(bytes, 0, samples, 0, bytes.length);
            return;
        }
        long maxValue = (long)Math.pow(2.0, bps0) - 1L;
        if (photoInterp == PhotoInterp.CMYK) {
            maxValue = Integer.MAX_VALUE;
        }
        if ((skipBits = (int)(8L - imageWidth * (long)bps0 * (long)nChannels % 8L)) == 8 || (long)(bytes.length * 8) < (long)bps0 * ((long)nChannels * imageWidth + imageHeight)) {
            skipBits = 0;
        }
        float lumaRed = 0.299f;
        float lumaGreen = 0.587f;
        float lumaBlue = 0.114f;
        int[] reference = ifd.getIFDIntArray(532);
        if (reference == null) {
            reference = new int[]{0, 0, 0, 0, 0, 0};
        }
        int[] subsampling = ifd.getIFDIntArray(530);
        TiffRational[] coefficients = (TiffRational[])ifd.getIFDValue(529);
        if (coefficients != null) {
            lumaRed = coefficients[0].floatValue();
            lumaGreen = coefficients[1].floatValue();
            lumaBlue = coefficients[2].floatValue();
        }
        int subX = subsampling == null ? 2 : subsampling[0];
        int subY = subsampling == null ? 2 : subsampling[1];
        int block = subX * subY;
        int nTiles = (int)(imageWidth / (long)subX);
        block0: for (int sample = 0; sample < sampleCount && (ndx = startIndex + sample) < nSamples; ++sample) {
            for (int channel = 0; channel < nChannels; ++channel) {
                int index = numBytes * (sample * nChannels + channel);
                int outputIndex = (channel * nSamples + ndx) * numBytes;
                if (photoInterp != PhotoInterp.Y_CB_CR) {
                    long value = 0L;
                    if (noDiv8) {
                        if (channel == 0 && photoInterp == PhotoInterp.RGB_PALETTE || photoInterp != PhotoInterp.CFA_ARRAY && photoInterp != PhotoInterp.RGB_PALETTE) {
                            value = bb.getBits(bps0) & 0xFFFF;
                            if ((long)ndx % imageWidth == imageWidth - 1L) {
                                bb.skipBits(skipBits);
                            }
                        }
                    } else {
                        value = DataTools.bytesToLong(bytes, index, numBytes, littleEndian);
                    }
                    if (photoInterp == PhotoInterp.WHITE_IS_ZERO || photoInterp == PhotoInterp.CMYK) {
                        value = maxValue - value;
                    }
                    if (outputIndex + numBytes > samples.length) continue;
                    DataTools.unpackBytes(value, samples, outputIndex, numBytes, littleEndian);
                    continue;
                }
                if (channel != nChannels - 1) continue;
                int lumaIndex = sample + 2 * (sample / block);
                int chromaIndex = sample / block * (block + 2) + block;
                if (chromaIndex + 1 >= bytes.length) continue block0;
                int tile = ndx / block;
                int pixel = ndx % block;
                long r = subY * (tile / nTiles) + pixel / subX;
                long c = subX * (tile % nTiles) + pixel % subX;
                int idx = (int)(r * imageWidth + c);
                if (idx >= nSamples) continue;
                int y = (bytes[lumaIndex] & 0xFF) - reference[0];
                int cb = (bytes[chromaIndex] & 0xFF) - reference[2];
                int cr = (bytes[chromaIndex + 1] & 0xFF) - reference[4];
                int red = (int)((float)cr * (2.0f - 2.0f * lumaRed) + (float)y);
                int blue = (int)((float)cb * (2.0f - 2.0f * lumaBlue) + (float)y);
                int green = (int)(((float)y - lumaBlue * (float)blue - lumaRed * (float)red) / lumaGreen);
                samples[idx] = (byte)(red & 0xFF);
                samples[nSamples + idx] = (byte)(green & 0xFF);
                samples[2 * nSamples + idx] = (byte)(blue & 0xFF);
            }
        }
    }

    long getNextOffset(long previous) throws IOException {
        if (this.bigTiff) {
            return this.in.readLong();
        }
        long offset = previous & 0xFFFFFFFF00000000L | (long)this.in.readInt() & 0xFFFFFFFFL;
        if (offset < previous && offset != 0L && this.in.length() > Integer.MAX_VALUE) {
            offset += 0x100000000L;
        }
        return offset;
    }

    TiffIFDEntry readTiffIFDEntry() throws IOException {
        int valueCount;
        int entryTag = this.in.readUnsignedShort();
        IFDType entryType = IFDType.get(this.in.readUnsignedShort());
        int n = valueCount = this.bigTiff ? (int)(this.in.readLong() & 0xFFFFFFFFFFFFFFFFL) : this.in.readInt();
        if (valueCount < 0) {
            throw new RuntimeException("Count of '" + valueCount + "' unexpected.");
        }
        int nValueBytes = valueCount * entryType.getBytesPerElement();
        int threshhold = this.bigTiff ? 8 : 4;
        long offset = nValueBytes > threshhold ? this.getNextOffset(0L) : this.in.getFilePointer();
        return new TiffIFDEntry(entryTag, entryType, valueCount, offset);
    }
}

