/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.in;

import java.io.File;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Vector;
import loci.common.DataTools;
import loci.common.DateTools;
import loci.common.Location;
import loci.common.RandomAccessInputStream;
import loci.formats.CoreMetadata;
import loci.formats.FormatException;
import loci.formats.FormatReader;
import loci.formats.FormatTools;
import loci.formats.MetadataTools;
import loci.formats.in.MDBParser;
import loci.formats.meta.FilterMetadata;
import loci.formats.meta.MetadataStore;
import loci.formats.tiff.IFD;
import loci.formats.tiff.IFDList;
import loci.formats.tiff.TiffParser;
import loci.formats.tiff.TiffTools;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class ZeissLSMReader
extends FormatReader {
    public static final String[] MDB_SUFFIX = new String[]{"mdb"};
    private static final int ZEISS_ID = 34412;
    private static final int TYPE_SUBBLOCK = 0;
    private static final int TYPE_ASCII = 2;
    private static final int TYPE_LONG = 4;
    private static final int TYPE_RATIONAL = 5;
    private static final int SUBBLOCK_RECORDING = 0x10000000;
    private static final int SUBBLOCK_LASER = 0x50000000;
    private static final int SUBBLOCK_TRACK = 0x40000000;
    private static final int SUBBLOCK_DETECTION_CHANNEL = 0x70000000;
    private static final int SUBBLOCK_ILLUMINATION_CHANNEL = -1879048192;
    private static final int SUBBLOCK_BEAM_SPLITTER = -1342177280;
    private static final int SUBBLOCK_DATA_CHANNEL = -805306368;
    private static final int SUBBLOCK_TIMER = 0x12000000;
    private static final int SUBBLOCK_MARKER = 0x14000000;
    private static final int SUBBLOCK_END = -1;
    private static final int RECORDING_NAME = 0x10000001;
    private static final int RECORDING_DESCRIPTION = 0x10000002;
    private static final int RECORDING_OBJECTIVE = 0x10000004;
    private static final int RECORDING_ZOOM = 0x10000016;
    private static final int RECORDING_SAMPLE_0TIME = 268435510;
    private static final int RECORDING_CAMERA_BINNING = 268435538;
    private static final int TRACK_ACQUIRE = 0x40000006;
    private static final int TRACK_TIME_BETWEEN_STACKS = 0x4000000B;
    private static final int LASER_NAME = 0x50000001;
    private static final int LASER_ACQUIRE = 0x50000002;
    private static final int LASER_POWER = 0x50000003;
    private static final int CHANNEL_DETECTOR_GAIN = 0x70000003;
    private static final int CHANNEL_PINHOLE_DIAMETER = 0x70000009;
    private static final int CHANNEL_AMPLIFIER_GAIN = 0x70000005;
    private static final int CHANNEL_FILTER_SET = 0x7000000F;
    private static final int CHANNEL_FILTER = 0x70000010;
    private static final int CHANNEL_ACQUIRE = 0x7000000B;
    private static final int CHANNEL_NAME = 1879048212;
    private static final int ILLUM_CHANNEL_ATTENUATION = -1879048190;
    private static final int ILLUM_CHANNEL_WAVELENGTH = -1879048189;
    private static final int ILLUM_CHANNEL_ACQUIRE = -1879048188;
    private static final int START_TIME = 268435510;
    private static final int DATA_CHANNEL_NAME = -805306367;
    private static final int DATA_CHANNEL_ACQUIRE = -805306345;
    private static final int BEAM_SPLITTER_FILTER = -1342177278;
    private static final int BEAM_SPLITTER_FILTER_SET = -1342177277;
    private static final int TEXT = 13;
    private static final int LINE = 14;
    private static final int SCALE_BAR = 15;
    private static final int OPEN_ARROW = 16;
    private static final int CLOSED_ARROW = 17;
    private static final int RECTANGLE = 18;
    private static final int ELLIPSE = 19;
    private static final int CLOSED_POLYLINE = 20;
    private static final int OPEN_POLYLINE = 21;
    private static final int CLOSED_BEZIER = 22;
    private static final int OPEN_BEZIER = 23;
    private static final int CIRCLE = 24;
    private static final int PALETTE = 25;
    private static final int POLYLINE_ARROW = 26;
    private static final int BEZIER_WITH_ARROW = 27;
    private static final int ANGLE = 28;
    private static final int CIRCLE_3POINT = 29;
    private static Hashtable<Integer, String> metadataKeys = ZeissLSMReader.createKeys();
    private double pixelSizeX;
    private double pixelSizeY;
    private double pixelSizeZ;
    private byte[][] lut = null;
    private Vector<Double> timestamps;
    private int validChannels;
    private String[] lsmFilenames;
    private Vector<IFDList> ifdsList;
    private TiffParser tiffParser;
    private int nextLaser = 0;
    private int nextDetector = 0;
    private int nextFilter = 0;
    private int nextFilterSet = 0;
    private int nextDataChannel = 0;
    private int nextIllumChannel = 0;
    private int nextDetectChannel = 0;
    private boolean splitPlanes = false;
    private float zoom;
    private Vector<String> imageNames;
    private String binning;

    public ZeissLSMReader() {
        super("Zeiss Laser-Scanning Microscopy", new String[]{"lsm", "mdb"});
    }

    @Override
    public void close(boolean fileOnly) throws IOException {
        super.close(fileOnly);
        if (!fileOnly) {
            this.pixelSizeZ = 0.0;
            this.pixelSizeY = 0.0;
            this.pixelSizeX = 0.0;
            this.lut = null;
            this.timestamps = null;
            this.validChannels = 0;
            this.lsmFilenames = null;
            this.ifdsList = null;
            this.tiffParser = null;
            this.nextDetector = 0;
            this.nextLaser = 0;
            this.nextFilterSet = 0;
            this.nextFilter = 0;
            this.nextDetectChannel = 0;
            this.nextIllumChannel = 0;
            this.nextDataChannel = 0;
            this.splitPlanes = false;
            this.zoom = 0.0f;
            this.imageNames = null;
            this.binning = null;
        }
    }

    @Override
    public boolean isThisType(RandomAccessInputStream stream) throws IOException {
        int blockLen = 4;
        if (!FormatTools.validStream(stream, 4, false)) {
            return false;
        }
        byte[] check = new byte[4];
        stream.readFully(check);
        return TiffTools.isValidHeader(check) || check[2] == 83 && check[3] == 116;
    }

    @Override
    public String[] getSeriesUsedFiles(boolean noPixels) {
        FormatTools.assertId(this.currentId, true, 1);
        if (noPixels) {
            if (ZeissLSMReader.checkSuffix(this.currentId, MDB_SUFFIX)) {
                return new String[]{this.currentId};
            }
            return null;
        }
        if (this.lsmFilenames == null) {
            return new String[]{this.currentId};
        }
        if (this.lsmFilenames.length == 1 && this.currentId.equals(this.lsmFilenames[0])) {
            return this.lsmFilenames;
        }
        return new String[]{this.lsmFilenames[this.getSeries()], this.currentId};
    }

    @Override
    public byte[][] get8BitLookupTable() throws FormatException, IOException {
        FormatTools.assertId(this.currentId, true, 1);
        if (this.lut == null || this.lut[this.getSeries()] == null || this.getPixelType() != 1) {
            return null;
        }
        byte[][] b = new byte[3][256];
        for (int i = 2; i >= 3 - this.validChannels; --i) {
            for (int j = 0; j < 256; ++j) {
                b[i][j] = (byte)j;
            }
        }
        return b;
    }

    @Override
    public short[][] get16BitLookupTable() throws FormatException, IOException {
        FormatTools.assertId(this.currentId, true, 1);
        if (this.lut == null || this.lut[this.getSeries()] == null || this.getPixelType() != 3) {
            return null;
        }
        short[][] s = new short[3][65536];
        for (int i = 2; i >= 3 - this.validChannels; --i) {
            for (int j = 0; j < s[i].length; ++j) {
                s[i][j] = (short)j;
            }
        }
        return s;
    }

    @Override
    public byte[] openBytes(int no, byte[] buf, int x, int y, int w, int h) throws FormatException, IOException {
        FormatTools.checkPlaneParameters(this, no, buf.length, x, y, w, h);
        this.in = new RandomAccessInputStream(this.lsmFilenames[this.getSeries()]);
        this.in.order(!this.isLittleEndian());
        this.tiffParser = new TiffParser(this.in);
        IFDList ifds = this.ifdsList.get(this.getSeries());
        if (this.splitPlanes && this.getSizeC() > 1) {
            int plane = no / this.getSizeC();
            int c = no % this.getSizeC();
            byte[][] samples = this.tiffParser.getSamples((IFD)ifds.get(plane), x, y, w, h);
            System.arraycopy(samples[c], 0, buf, 0, buf.length);
        } else {
            this.tiffParser.getSamples((IFD)ifds.get(no), buf, x, y, w, h);
        }
        this.in.close();
        return buf;
    }

    @Override
    protected void initFile(String id) throws FormatException, IOException {
        int series;
        this.debug("ZeissLSMReader.initFile(" + id + ")");
        super.initFile(id);
        if (!ZeissLSMReader.checkSuffix(id, MDB_SUFFIX) && this.isGroupFiles()) {
            Location parentFile = new Location(id).getAbsoluteFile().getParentFile();
            String[] fileList = parentFile.list();
            for (int i = 0; i < fileList.length; ++i) {
                String[] lsms;
                Location file;
                if (fileList[i].startsWith(".") || !ZeissLSMReader.checkSuffix(fileList[i], MDB_SUFFIX) || (file = new Location(parentFile, fileList[i]).getAbsoluteFile()).isDirectory()) continue;
                for (String lsm : lsms = this.parseMDB(file.getAbsolutePath())) {
                    if (!id.endsWith(lsm) && !lsm.endsWith(id)) continue;
                    this.setId(file.getAbsolutePath());
                    return;
                }
            }
            this.lsmFilenames = new String[]{id};
        } else {
            this.lsmFilenames = ZeissLSMReader.checkSuffix(id, MDB_SUFFIX) ? this.parseMDB(id) : new String[]{id};
        }
        if (this.lsmFilenames.length == 0) {
            throw new FormatException("LSM files were not found.");
        }
        this.timestamps = new Vector();
        this.imageNames = new Vector();
        this.core = new CoreMetadata[this.lsmFilenames.length];
        this.ifdsList = new Vector();
        this.ifdsList.setSize(this.core.length);
        for (int i = 0; i < this.core.length; ++i) {
            this.core[i] = new CoreMetadata();
            RandomAccessInputStream s = new RandomAccessInputStream(this.lsmFilenames[i]);
            this.core[i].littleEndian = s.read() == 73;
            s.order(this.isLittleEndian());
            s.seek(0L);
            this.ifdsList.set(i, new TiffParser(s).getIFDs());
            s.close();
        }
        this.status("Removing thumbnails");
        FilterMetadata store = new FilterMetadata(this.getMetadataStore(), this.isMetadataFiltered());
        for (series = 0; series < this.ifdsList.size(); ++series) {
            int i;
            IFDList ifds = this.ifdsList.get(series);
            IFDList newIFDs = new IFDList();
            for (i = 0; i < ifds.size(); ++i) {
                IFD ifd = (IFD)ifds.get(i);
                long subFileType = ifd.getIFDLongValue(254, false, 0L);
                if (subFileType != 0L) continue;
                if (ifd.getCompression() != 5) {
                    ifd.put(new Integer(317), new Integer(1));
                }
                newIFDs.add(ifd);
            }
            this.ifdsList.set(series, newIFDs);
            ifds = newIFDs;
            for (i = 1; i < ifds.size(); ++i) {
                long thisOffset = ((IFD)ifds.get(i)).getStripOffsets()[0] & 0xFFFFFFFFL;
                long prevOffset = ((IFD)ifds.get(i - 1)).getStripOffsets()[0];
                if (prevOffset < 0L) {
                    prevOffset &= 0xFFFFFFFFL;
                }
                if (prevOffset <= thisOffset) continue;
                ((IFD)ifds.get(i)).put(new Integer(273), new Long(thisOffset += 0xFFFFFFFFL));
            }
            this.initMetadata(series);
            this.core[series].littleEndian = !this.isLittleEndian();
            store.setPixelsBigEndian(new Boolean(!this.isLittleEndian()), series, 0);
        }
        for (series = 0; series < this.ifdsList.size(); ++series) {
            store.setImageName(this.imageNames.get(series), series);
        }
        this.setSeries(0);
    }

    protected void initMetadata(int series) throws FormatException, IOException {
        int nLogicalChannels;
        this.setSeries(series);
        IFDList ifds = this.ifdsList.get(series);
        IFD ifd = (IFD)ifds.get(0);
        this.in = new RandomAccessInputStream(this.lsmFilenames[series]);
        this.in.order(this.isLittleEndian());
        this.tiffParser = new TiffParser(this.in);
        int photo = ifd.getPhotometricInterpretation();
        int samples = ifd.getSamplesPerPixel();
        this.core[series].sizeX = (int)ifd.getImageWidth();
        this.core[series].sizeY = (int)ifd.getImageLength();
        this.core[series].rgb = samples > 1 || photo == 2;
        this.core[series].interleaved = false;
        this.core[series].sizeC = this.isRGB() ? samples : 1;
        this.core[series].pixelType = ifd.getPixelType();
        this.core[series].imageCount = ifds.size();
        this.core[series].sizeZ = this.getImageCount();
        this.core[series].sizeT = 1;
        this.status("Reading LSM metadata for series #" + series);
        FilterMetadata store = new FilterMetadata(this.getMetadataStore(), this.isMetadataFiltered());
        String imageName = this.lsmFilenames[series];
        if (imageName.indexOf(".") != -1) {
            imageName = imageName.substring(0, imageName.lastIndexOf("."));
        }
        if (imageName.indexOf(File.separator) != -1) {
            imageName = imageName.substring(imageName.lastIndexOf(File.separator) + 1);
        }
        String instrumentID = MetadataTools.createLSID("Instrument", series);
        store.setInstrumentID(instrumentID, series);
        store.setImageInstrumentRef(instrumentID, series);
        short[] s = ifd.getIFDShortArray(34412, true);
        byte[] cz = new byte[s.length];
        for (int i = 0; i < s.length; ++i) {
            cz[i] = (byte)s[i];
        }
        RandomAccessInputStream ras = new RandomAccessInputStream(cz);
        ras.order(this.isLittleEndian());
        this.addSeriesMeta("MagicNumber ", ras.readInt());
        this.addSeriesMeta("StructureSize", ras.readInt());
        this.addSeriesMeta("DimensionX", ras.readInt());
        this.addSeriesMeta("DimensionY", ras.readInt());
        this.core[series].sizeZ = ras.readInt();
        ras.skipBytes(4);
        this.core[series].sizeT = ras.readInt();
        int dataType = ras.readInt();
        switch (dataType) {
            case 2: {
                this.addSeriesMeta("DataType", "12 bit unsigned integer");
                break;
            }
            case 5: {
                this.addSeriesMeta("DataType", "32 bit float");
                break;
            }
            case 0: {
                this.addSeriesMeta("DataType", "varying data types");
                break;
            }
            default: {
                this.addSeriesMeta("DataType", "8 bit unsigned integer");
            }
        }
        this.addSeriesMeta("ThumbnailX", ras.readInt());
        this.addSeriesMeta("ThumbnailY", ras.readInt());
        this.pixelSizeX = ras.readDouble() * 1000000.0;
        this.pixelSizeY = ras.readDouble() * 1000000.0;
        this.pixelSizeZ = ras.readDouble() * 1000000.0;
        this.addSeriesMeta("VoxelSizeX", new Double(this.pixelSizeX));
        this.addSeriesMeta("VoxelSizeY", new Double(this.pixelSizeY));
        this.addSeriesMeta("VoxelSizeZ", new Double(this.pixelSizeZ));
        this.addSeriesMeta("OriginX", ras.readDouble());
        this.addSeriesMeta("OriginY", ras.readDouble());
        this.addSeriesMeta("OriginZ", ras.readDouble());
        short scanType = ras.readShort();
        switch (scanType) {
            case 0: {
                this.addSeriesMeta("ScanType", "x-y-z scan");
                this.core[series].dimensionOrder = "XYZCT";
                break;
            }
            case 1: {
                this.addSeriesMeta("ScanType", "z scan (x-z plane)");
                this.core[series].dimensionOrder = "XYZCT";
                break;
            }
            case 2: {
                this.addSeriesMeta("ScanType", "line scan");
                this.core[series].dimensionOrder = "XYZCT";
                break;
            }
            case 3: {
                this.addSeriesMeta("ScanType", "time series x-y");
                this.core[series].dimensionOrder = "XYTCZ";
                break;
            }
            case 4: {
                this.addSeriesMeta("ScanType", "time series x-z");
                this.core[series].dimensionOrder = "XYZTC";
                break;
            }
            case 5: {
                this.addSeriesMeta("ScanType", "time series 'Mean of ROIs'");
                this.core[series].dimensionOrder = "XYTCZ";
                break;
            }
            case 6: {
                this.addSeriesMeta("ScanType", "time series x-y-z");
                this.core[series].dimensionOrder = "XYZTC";
                break;
            }
            case 7: {
                this.addSeriesMeta("ScanType", "spline scan");
                this.core[series].dimensionOrder = "XYCTZ";
                break;
            }
            case 8: {
                this.addSeriesMeta("ScanType", "spline scan x-z");
                this.core[series].dimensionOrder = "XYCZT";
                break;
            }
            case 9: {
                this.addSeriesMeta("ScanType", "time series spline plane x-z");
                this.core[series].dimensionOrder = "XYTCZ";
                break;
            }
            case 10: {
                this.addSeriesMeta("ScanType", "point mode");
                this.core[series].dimensionOrder = "XYZCT";
                break;
            }
            default: {
                this.addSeriesMeta("ScanType", "x-y-z scan");
                this.core[series].dimensionOrder = "XYZCT";
            }
        }
        boolean bl = this.core[series].indexed = this.lut != null && this.lut[series] != null && this.getSizeC() == 1;
        if (this.isIndexed()) {
            this.core[series].sizeC = 1;
            this.core[series].rgb = false;
        }
        if (this.getSizeC() == 0) {
            this.core[series].sizeC = 1;
        }
        if (this.isRGB()) {
            this.core[series].dimensionOrder = this.getDimensionOrder().replaceAll("C", "");
            this.core[series].dimensionOrder = this.getDimensionOrder().replaceAll("XY", "XYC");
        }
        if (this.isIndexed()) {
            this.core[series].rgb = false;
        }
        this.core[series].imageCount = this.getEffectiveSizeC() == 0 ? this.getSizeZ() * this.getSizeT() : this.getSizeZ() * this.getSizeT() * this.getEffectiveSizeC();
        if (this.getImageCount() != ifds.size()) {
            int diff = this.getImageCount() - ifds.size();
            this.core[series].imageCount = ifds.size();
            if (diff % this.getSizeZ() == 0) {
                this.core[series].sizeT -= diff / this.getSizeZ();
            } else if (diff % this.getSizeT() == 0) {
                this.core[series].sizeZ -= diff / this.getSizeT();
            } else if (this.getSizeZ() > 1) {
                this.core[series].sizeZ = ifds.size();
                this.core[series].sizeT = 1;
            } else if (this.getSizeT() > 1) {
                this.core[series].sizeT = ifds.size();
                this.core[series].sizeZ = 1;
            }
        }
        if (this.getSizeZ() == 0) {
            this.core[series].sizeZ = this.getImageCount();
        }
        if (this.getSizeT() == 0) {
            this.core[series].sizeT = this.getImageCount() / this.getSizeZ();
        }
        MetadataTools.setDefaultCreationDate(store, this.getCurrentFile(), series);
        short spectralScan = ras.readShort();
        if (spectralScan != 1) {
            this.addSeriesMeta("SpectralScan", "no spectral scan");
        } else {
            this.addSeriesMeta("SpectralScan", "acquired with spectral scan");
        }
        int type = ras.readInt();
        switch (type) {
            case 1: {
                this.addSeriesMeta("DataType2", "calculated data");
                break;
            }
            case 2: {
                this.addSeriesMeta("DataType2", "animation");
                break;
            }
            default: {
                this.addSeriesMeta("DataType2", "original scan data");
            }
        }
        long[] overlayOffsets = new long[9];
        String[] overlayKeys = new String[]{"VectorOverlay", "InputLut", "OutputLut", "ROI", "BleachROI", "MeanOfRoisOverlay", "TopoIsolineOverlay", "TopoProfileOverlay", "LinescanOverlay"};
        overlayOffsets[0] = ras.readInt();
        overlayOffsets[1] = ras.readInt();
        overlayOffsets[2] = ras.readInt();
        long channelColorsOffset = ras.readInt();
        this.addSeriesMeta("TimeInterval", ras.readDouble());
        ras.skipBytes(4);
        long scanInformationOffset = ras.readInt();
        ras.skipBytes(4);
        long timeStampOffset = ras.readInt();
        long eventListOffset = ras.readInt();
        overlayOffsets[3] = ras.readInt();
        overlayOffsets[4] = ras.readInt();
        ras.skipBytes(4);
        this.addSeriesMeta("DisplayAspectX", ras.readDouble());
        this.addSeriesMeta("DisplayAspectY", ras.readDouble());
        this.addSeriesMeta("DisplayAspectZ", ras.readDouble());
        this.addSeriesMeta("DisplayAspectTime", ras.readDouble());
        overlayOffsets[5] = ras.readInt();
        overlayOffsets[6] = ras.readInt();
        overlayOffsets[7] = ras.readInt();
        overlayOffsets[8] = ras.readInt();
        for (int i = 0; i < overlayOffsets.length; ++i) {
            this.parseOverlays(series, overlayOffsets[i], overlayKeys[i], store);
        }
        this.addSeriesMeta("ToolbarFlags", ras.readInt());
        int wavelengthOffset = ras.readInt();
        ras.close();
        this.addSeriesMeta("DimensionZ", this.getSizeZ());
        this.addSeriesMeta("DimensionChannels", this.getSizeC());
        if (channelColorsOffset != 0L) {
            this.in.seek(channelColorsOffset + 16L);
            int namesOffset = this.in.readInt();
            if (namesOffset > 0) {
                this.in.skipBytes(namesOffset - 16);
                for (int i = 0; i < this.getSizeC() && this.in.getFilePointer() < this.in.length() - 1L; ++i) {
                    String name = this.in.readCString();
                    if (name.length() > 128) continue;
                    this.addSeriesMeta("ChannelName" + i, name);
                }
            }
        }
        if (timeStampOffset != 0L) {
            this.in.seek(timeStampOffset + 8L);
            for (int i = 0; i < this.getSizeT(); ++i) {
                double stamp = this.in.readDouble();
                this.addSeriesMeta("TimeStamp" + i, stamp);
                this.timestamps.add(new Double(stamp));
            }
        }
        if (eventListOffset != 0L) {
            this.in.seek(eventListOffset + 4L);
            int numEvents = this.in.readInt();
            this.in.seek(this.in.getFilePointer() - 4L);
            this.in.order(!this.in.isLittleEndian());
            int tmpEvents = this.in.readInt();
            numEvents = numEvents < 0 ? tmpEvents : Math.min(numEvents, tmpEvents);
            this.in.order(!this.in.isLittleEndian());
            if (numEvents > 65535) {
                numEvents = 0;
            }
            for (int i = 0; i < numEvents; ++i) {
                if (this.in.getFilePointer() + 16L > this.in.length()) continue;
                int size = this.in.readInt();
                double eventTime = this.in.readDouble();
                int eventType = this.in.readInt();
                this.addSeriesMeta("Event" + i + " Time", eventTime);
                this.addSeriesMeta("Event" + i + " Type", eventType);
                long fp = this.in.getFilePointer();
                int len = size - 16;
                if (len > 65536) {
                    len = 65536;
                }
                if (len < 0) {
                    len = 0;
                }
                this.addSeriesMeta("Event" + i + " Description", this.in.readString(len));
                this.in.seek(fp + (long)size - 16L);
                if (this.in.getFilePointer() < 0L) break;
            }
        }
        if (scanInformationOffset != 0L) {
            SubBlock[] metadataBlocks;
            this.in.seek(scanInformationOffset);
            this.nextDetector = 0;
            this.nextLaser = 0;
            this.nextFilterSet = 0;
            this.nextFilter = 0;
            this.nextIllumChannel = 0;
            this.nextDetectChannel = 0;
            this.nextDataChannel = 0;
            Vector<Recording> blocks = new Vector<Recording>();
            while (this.in.getFilePointer() < this.in.length() - 12L && this.in.getFilePointer() >= 0L) {
                int entry = this.in.readInt();
                int blockType = this.in.readInt();
                int dataSize = this.in.readInt();
                if (blockType == 0) {
                    SubBlock block = null;
                    switch (entry) {
                        case 0x10000000: {
                            block = new Recording();
                            break;
                        }
                        case 0x50000000: {
                            block = new Laser();
                            break;
                        }
                        case 0x40000000: {
                            block = new Track();
                            break;
                        }
                        case 0x70000000: {
                            block = new DetectionChannel();
                            break;
                        }
                        case -1879048192: {
                            block = new IlluminationChannel();
                            break;
                        }
                        case -1342177280: {
                            block = new BeamSplitter();
                            break;
                        }
                        case -805306368: {
                            block = new DataChannel();
                            break;
                        }
                        case 0x12000000: {
                            block = new Timer();
                            break;
                        }
                        case 0x14000000: {
                            block = new Marker();
                        }
                    }
                    if (block == null) continue;
                    blocks.add((Recording)block);
                    continue;
                }
                this.in.skipBytes(dataSize);
            }
            Vector<SubBlock> nonAcquiredBlocks = new Vector<SubBlock>();
            for (SubBlock block : metadataBlocks = blocks.toArray(new SubBlock[0])) {
                block.addToHashtable();
                if (block.acquire) continue;
                nonAcquiredBlocks.add(block);
                blocks.remove(block);
            }
            for (int i = 0; i < blocks.size(); ++i) {
                SubBlock prevBlock;
                SubBlock block = (SubBlock)blocks.get(i);
                if (block instanceof IlluminationChannel && i < blocks.size() - 1) {
                    SubBlock nextBlock = (SubBlock)blocks.get(i + 1);
                    if (!(nextBlock instanceof DataChannel) && !(nextBlock instanceof IlluminationChannel)) {
                        ((IlluminationChannel)block).wavelength = null;
                    }
                } else if (block instanceof DetectionChannel && i > 0 && !((prevBlock = (SubBlock)blocks.get(i - 1)) instanceof Track) && !(prevBlock instanceof DetectionChannel)) {
                    block.acquire = false;
                    nonAcquiredBlocks.add(block);
                }
                if (!block.acquire) continue;
                this.populateMetadataStore(block, store, series);
            }
            for (SubBlock block : nonAcquiredBlocks) {
                this.populateMetadataStore(block, store, series);
            }
        }
        int n = nLogicalChannels = this.nextDataChannel == 0 ? 1 : this.nextDataChannel;
        if (nLogicalChannels == this.getSizeC()) {
            this.splitPlanes = this.isRGB();
            this.core[series].rgb = false;
            if (this.splitPlanes) {
                this.core[series].imageCount *= this.getSizeC();
            }
        }
        MetadataTools.populatePixels(store, this, true);
        this.imageNames.add(imageName);
        Float pixX = new Float((float)this.pixelSizeX);
        Float pixY = new Float((float)this.pixelSizeY);
        Float pixZ = new Float((float)this.pixelSizeZ);
        store.setDimensionsPhysicalSizeX(pixX, series, 0);
        store.setDimensionsPhysicalSizeY(pixY, series, 0);
        store.setDimensionsPhysicalSizeZ(pixZ, series, 0);
        float firstStamp = 0.0f;
        if (this.timestamps.size() > 0) {
            firstStamp = this.timestamps.get(0).floatValue();
        }
        for (int i = 0; i < this.getImageCount(); ++i) {
            float nextStamp;
            int[] zct = FormatTools.getZCTCoords(this, i);
            if (zct[2] >= this.timestamps.size()) continue;
            float thisStamp = this.timestamps.get(zct[2]).floatValue();
            store.setPlaneTimingDeltaT(new Float(thisStamp - firstStamp), series, 0, i);
            int index = zct[2] + 1;
            float f = nextStamp = index < this.timestamps.size() ? this.timestamps.get(index).floatValue() : thisStamp;
            if (i == this.getSizeT() - 1 && zct[2] > 0) {
                thisStamp = this.timestamps.get(zct[2] - 1).floatValue();
            }
            store.setPlaneTimingExposureTime(new Float(nextStamp - thisStamp), series, 0, i);
        }
        this.in.close();
    }

    protected void populateMetadataStore(SubBlock block, MetadataStore store, int series) {
        if (block instanceof Recording) {
            Recording recording = (Recording)block;
            String objectiveID = MetadataTools.createLSID("Objective", series, 0);
            if (recording.acquire) {
                store.setImageDescription(recording.description, series);
                store.setImageCreationDate(recording.startTime, series);
                store.setObjectiveSettingsObjective(objectiveID, series);
                this.binning = recording.binning;
            }
            store.setObjectiveCorrection(recording.correction, series, 0);
            store.setObjectiveImmersion(recording.immersion, series, 0);
            store.setObjectiveNominalMagnification(recording.magnification, series, 0);
            store.setObjectiveLensNA(recording.lensNA, series, 0);
            store.setObjectiveIris(recording.iris, series, 0);
            store.setObjectiveID(objectiveID, series, 0);
        } else if (block instanceof Laser) {
            Laser laser = (Laser)block;
            if (laser.medium != null) {
                store.setLaserLaserMedium(laser.medium, series, this.nextLaser);
            }
            if (laser.type != null) {
                store.setLaserType(laser.type, series, this.nextLaser);
            }
            String lightSourceID = MetadataTools.createLSID("LightSource", series, this.nextLaser);
            store.setLightSourceID(lightSourceID, series, this.nextLaser);
            ++this.nextLaser;
        } else if (block instanceof Track) {
            Track track = (Track)block;
            if (track.acquire) {
                store.setDimensionsTimeIncrement(track.timeIncrement, series, 0);
            }
        } else if (block instanceof DataChannel) {
            DataChannel channel = (DataChannel)block;
            if (channel.name != null && this.nextDataChannel < this.getSizeC() && channel.acquire) {
                store.setLogicalChannelName(channel.name, series, this.nextDataChannel++);
            }
        } else if (block instanceof DetectionChannel) {
            DetectionChannel channel = (DetectionChannel)block;
            if (channel.pinhole != null && channel.pinhole.floatValue() != 0.0f && this.nextDetectChannel < this.getSizeC() && channel.acquire) {
                store.setLogicalChannelPinholeSize(channel.pinhole, series, this.nextDetectChannel);
            }
            if (channel.filter != null) {
                String id = MetadataTools.createLSID("Filter", series, this.nextFilter);
                if (channel.acquire && this.nextDetectChannel < this.getSizeC()) {
                    store.setLogicalChannelSecondaryEmissionFilter(id, series, this.nextDetectChannel);
                }
                store.setFilterID(id, series, this.nextFilter);
                store.setFilterModel(channel.filter, series, this.nextFilter);
                int space = channel.filter.indexOf(" ");
                if (space != -1) {
                    String type = channel.filter.substring(0, space).trim();
                    if (type.equals("BP")) {
                        type = "BandPass";
                    } else if (type.equals("LP")) {
                        type = "LongPass";
                    }
                    store.setFilterType(type, series, this.nextFilter);
                    String transmittance = channel.filter.substring(space + 1).trim();
                    String[] v = transmittance.split("-");
                    try {
                        store.setTransmittanceRangeCutIn(new Integer(v[0].trim()), series, this.nextFilter);
                    }
                    catch (NumberFormatException e) {
                        // empty catch block
                    }
                    if (v.length > 1) {
                        try {
                            store.setTransmittanceRangeCutOut(new Integer(v[1].trim()), series, this.nextFilter);
                        }
                        catch (NumberFormatException e) {
                            // empty catch block
                        }
                    }
                }
                ++this.nextFilter;
            }
            if (channel.channelName != null) {
                String detectorID = MetadataTools.createLSID("Detector", series, this.nextDetector);
                store.setDetectorID(detectorID, series, this.nextDetector);
                if (channel.acquire && this.nextDetector < this.getSizeC()) {
                    store.setDetectorSettingsDetector(detectorID, series, this.nextDetector);
                    store.setDetectorSettingsBinning(this.binning, series, this.nextDetector);
                }
            }
            if (channel.amplificationGain != null) {
                store.setDetectorAmplificationGain(channel.amplificationGain, series, this.nextDetector);
            }
            if (channel.gain != null) {
                store.setDetectorGain(channel.gain, series, this.nextDetector);
            }
            store.setDetectorType("Unknown", series, this.nextDetector);
            store.setDetectorZoom(new Float(this.zoom), series, this.nextDetector);
            ++this.nextDetectChannel;
            ++this.nextDetector;
        } else if (block instanceof IlluminationChannel) {
            IlluminationChannel channel = (IlluminationChannel)block;
            boolean mustIncrement = false;
            if (this.nextIllumChannel < this.getSizeC()) {
                if (channel.wavelength != null && channel.acquire) {
                    store.setLogicalChannelExWave(channel.wavelength, series, this.nextIllumChannel);
                    mustIncrement = true;
                }
                if (mustIncrement || this.nextIllumChannel < this.getSizeC() - 1) {
                    ++this.nextIllumChannel;
                }
            }
        } else if (block instanceof BeamSplitter) {
            BeamSplitter beamSplitter = (BeamSplitter)block;
            if (beamSplitter.filterSet != null) {
                String filterSetID = MetadataTools.createLSID("FilterSet", series, this.nextFilterSet);
                store.setFilterSetID(filterSetID, series, this.nextFilterSet);
                if (beamSplitter.filter != null) {
                    String id = MetadataTools.createLSID("Dichroic", series, this.nextFilter);
                    store.setDichroicID(id, series, this.nextFilter);
                    store.setDichroicModel(beamSplitter.filter, series, this.nextFilter);
                    store.setFilterSetDichroic(id, series, this.nextFilterSet);
                    ++this.nextFilter;
                }
                ++this.nextFilterSet;
            }
        }
    }

    protected void parseOverlays(int series, long data, String suffix, MetadataStore store) throws IOException {
        if (data == 0L) {
            return;
        }
        String prefix = "Series " + series + " ";
        this.in.seek(data);
        int numberOfShapes = this.in.readInt();
        int size = this.in.readInt();
        if (size <= 194) {
            return;
        }
        this.in.skipBytes(20);
        boolean valid = this.in.readInt() == 1;
        this.in.skipBytes(164);
        for (int i = 0; i < numberOfShapes; ++i) {
            long offset = this.in.getFilePointer();
            int type = this.in.readInt();
            int blockLength = this.in.readInt();
            int lineWidth = this.in.readInt();
            int measurements = this.in.readInt();
            double textOffsetX = this.in.readDouble();
            double textOffsetY = this.in.readDouble();
            int color = this.in.readInt();
            boolean validShape = this.in.readInt() != 0;
            int knotWidth = this.in.readInt();
            int catchArea = this.in.readInt();
            int fontHeight = this.in.readInt();
            int fontWidth = this.in.readInt();
            int fontEscapement = this.in.readInt();
            int fontOrientation = this.in.readInt();
            int fontWeight = this.in.readInt();
            boolean fontItalic = this.in.readInt() != 0;
            boolean fontUnderlined = this.in.readInt() != 0;
            boolean fontStrikeout = this.in.readInt() != 0;
            int fontCharSet = this.in.readInt();
            int fontOutputPrecision = this.in.readInt();
            int fontClipPrecision = this.in.readInt();
            int fontQuality = this.in.readInt();
            int fontPitchAndFamily = this.in.readInt();
            String fontName = DataTools.stripString(this.in.readString(64));
            boolean enabled = this.in.readShort() == 0;
            boolean moveable = this.in.readInt() == 0;
            this.in.skipBytes(34);
            store.setShapeFontFamily(fontName, series, 0, i);
            store.setShapeFontSize(new Integer(fontHeight), series, 0, i);
            store.setShapeFontStyle(fontItalic ? "normal" : "italic", series, 0, i);
            store.setShapeFontWeight(String.valueOf(fontWeight), series, 0, i);
            store.setShapeLocked(new Boolean(moveable), series, 0, i);
            store.setShapeStrokeColor(String.valueOf(color), series, 0, i);
            store.setShapeStrokeWidth(new Integer(lineWidth), series, 0, i);
            store.setShapeTextDecoration(fontUnderlined ? "underline" : (fontStrikeout ? "line-through" : "normal"), series, 0, i);
            store.setShapeVisibility(new Boolean(enabled), series, 0, i);
            switch (type) {
                case 13: {
                    double x = this.in.readDouble();
                    double y = this.in.readDouble();
                    String text = DataTools.stripString(this.in.readCString());
                    store.setShapeText(text, series, 0, i);
                    break;
                }
                case 14: {
                    this.in.skipBytes(4);
                    double startX = this.in.readDouble();
                    double startY = this.in.readDouble();
                    double endX = this.in.readDouble();
                    double endY = this.in.readDouble();
                    store.setLineX1(String.valueOf(startX), series, 0, i);
                    store.setLineY1(String.valueOf(startY), series, 0, i);
                    store.setLineX2(String.valueOf(endX), series, 0, i);
                    store.setLineY2(String.valueOf(endY), series, 0, i);
                    break;
                }
                case 15: 
                case 16: 
                case 17: 
                case 25: {
                    this.in.skipBytes(36);
                    break;
                }
                case 18: {
                    this.in.skipBytes(4);
                    double topX = this.in.readDouble();
                    double topY = this.in.readDouble();
                    double bottomX = this.in.readDouble();
                    double bottomY = this.in.readDouble();
                    double width = Math.abs(bottomX - topX);
                    double height = Math.abs(bottomY - topY);
                    topX = Math.min(topX, bottomX);
                    topY = Math.min(topY, bottomY);
                    store.setRectX(String.valueOf(topX), series, 0, i);
                    store.setRectY(String.valueOf(topY), series, 0, i);
                    store.setRectWidth(String.valueOf(width), series, 0, i);
                    store.setRectHeight(String.valueOf(height), series, 0, i);
                    break;
                }
                case 19: {
                    int knots = this.in.readInt();
                    double[] xs = new double[knots];
                    double[] ys = new double[knots];
                    for (int j = 0; j < xs.length; ++j) {
                        xs[j] = this.in.readDouble();
                        ys[j] = this.in.readDouble();
                    }
                    double rx = 0.0;
                    double ry = 0.0;
                    double centerX = 0.0;
                    double centerY = 0.0;
                    if (knots == 4) {
                        double r1x = Math.abs(xs[2] - xs[0]) / 2.0;
                        double r1y = Math.abs(ys[2] - ys[0]) / 2.0;
                        double r2x = Math.abs(xs[3] - xs[1]) / 2.0;
                        double r2y = Math.abs(ys[3] - ys[1]) / 2.0;
                        if (r1x > r2x) {
                            ry = r1y;
                            rx = r2x;
                            centerX = Math.min(xs[3], xs[1]) + rx;
                            centerY = Math.min(ys[2], ys[0]) + ry;
                        } else {
                            ry = r2y;
                            rx = r1x;
                            centerX = Math.min(xs[2], xs[0]) + rx;
                            centerY = Math.min(ys[3], ys[1]) + ry;
                        }
                    } else if (knots == 3) {
                        centerX = xs[0];
                        centerY = ys[0];
                        rx = Math.sqrt(Math.pow(xs[1] - xs[0], 2.0) + Math.pow(ys[1] - ys[0], 2.0));
                        ry = Math.sqrt(Math.pow(xs[2] - xs[0], 2.0) + Math.pow(ys[2] - ys[0], 2.0));
                        double slope = (ys[2] - centerY) / (xs[2] - centerX);
                        double theta = Math.toDegrees(Math.atan(slope));
                        store.setEllipseTransform("rotate(" + theta + " " + centerX + " " + centerY + ")", series, 0, i);
                    }
                    store.setEllipseCx(String.valueOf(centerX), series, 0, i);
                    store.setEllipseCy(String.valueOf(centerY), series, 0, i);
                    store.setEllipseRx(String.valueOf(rx), series, 0, i);
                    store.setEllipseRy(String.valueOf(ry), series, 0, i);
                    break;
                }
                case 24: {
                    this.in.skipBytes(4);
                    double centerX = this.in.readDouble();
                    double centerY = this.in.readDouble();
                    double curveX = this.in.readDouble();
                    double curveY = this.in.readDouble();
                    double radius = Math.sqrt(Math.pow(curveX - centerX, 2.0) + Math.pow(curveY - centerY, 2.0));
                    store.setCircleCx(String.valueOf(centerX), series, 0, i);
                    store.setCircleCy(String.valueOf(centerY), series, 0, i);
                    store.setCircleR(String.valueOf(radius), series, 0, i);
                    break;
                }
                case 29: {
                    this.in.skipBytes(4);
                    double[][] points = new double[3][2];
                    for (int j = 0; j < points.length; ++j) {
                        for (int k = 0; k < points[j].length; ++k) {
                            points[j][k] = this.in.readDouble();
                        }
                    }
                    double s = 0.5 * ((points[1][0] - points[2][0]) * (points[0][0] - points[2][0]) - (points[1][1] - points[2][1]) * (points[2][1] - points[0][1]));
                    double div = (points[0][0] - points[1][0]) * (points[2][1] - points[0][1]) - (points[1][1] - points[0][1]) * (points[0][0] - points[2][0]);
                    double cx = 0.5 * (points[0][0] + points[1][0]) + (s /= div) * (points[1][1] - points[0][1]);
                    double cy = 0.5 * (points[0][1] + points[1][1]) + s * (points[0][0] - points[1][0]);
                    double r = Math.sqrt(Math.pow(points[0][0] - cx, 2.0) + Math.pow(points[0][1] - cy, 2.0));
                    store.setCircleCx(String.valueOf(cx), series, 0, i);
                    store.setCircleCy(String.valueOf(cy), series, 0, i);
                    store.setCircleR(String.valueOf(r), series, 0, i);
                    break;
                }
                case 28: {
                    this.in.skipBytes(4);
                    double[][] points = new double[3][2];
                    for (int j = 0; j < points.length; ++j) {
                        for (int k = 0; k < points[j].length; ++k) {
                            points[j][k] = this.in.readDouble();
                        }
                    }
                    StringBuffer p = new StringBuffer();
                    for (int j = 0; j < points.length; ++j) {
                        p.append(points[j][0]);
                        p.append(",");
                        p.append(points[j][1]);
                        if (j >= points.length - 1) continue;
                        p.append(" ");
                    }
                    store.setPolylinePoints(p.toString(), series, 0, i);
                    break;
                }
                case 20: 
                case 21: 
                case 26: {
                    int k;
                    int j;
                    int nKnots = this.in.readInt();
                    double[][] points = new double[nKnots][2];
                    for (j = 0; j < points.length; ++j) {
                        for (k = 0; k < points[j].length; ++k) {
                            points[j][k] = this.in.readDouble();
                        }
                    }
                    StringBuffer p = new StringBuffer();
                    for (j = 0; j < points.length; ++j) {
                        p.append(points[j][0]);
                        p.append(",");
                        p.append(points[j][1]);
                        if (j >= points.length - 1) continue;
                        p.append(" ");
                    }
                    if (type == 20) {
                        store.setPolygonPoints(p.toString(), series, 0, i);
                        break;
                    }
                    store.setPolylinePoints(p.toString(), series, 0, i);
                    break;
                }
                case 22: 
                case 23: 
                case 27: {
                    int k;
                    int j;
                    int nKnots = this.in.readInt();
                    double[][] points = new double[nKnots][2];
                    for (j = 0; j < points.length; ++j) {
                        for (k = 0; k < points[j].length; ++k) {
                            points[j][k] = this.in.readDouble();
                        }
                    }
                    break;
                }
            }
            this.in.seek(offset + (long)blockLength);
        }
    }

    private String[] parseMDB(String mdbFile) throws FormatException {
        Location mdb = new Location(mdbFile).getAbsoluteFile();
        Location parent = mdb.getParentFile();
        Vector<String[]>[] tables = MDBParser.parseDatabase(mdbFile);
        Vector<String> referencedLSMs = new Vector<String>();
        for (int table = 0; table < tables.length; ++table) {
            String[] columnNames = tables[table].get(0);
            String tableName = columnNames[0];
            for (int row = 1; row < tables[table].size(); ++row) {
                String[] tableRow = tables[table].get(row);
                for (int col = 0; col < tableRow.length; ++col) {
                    Location file;
                    String key = tableName + " " + columnNames[col + 1] + " " + row;
                    this.addGlobalMeta(key, tableRow[col]);
                    if (!tableName.equals("Recordings") || columnNames[col + 1] == null || !columnNames[col + 1].equals("SampleData")) continue;
                    String filename = tableRow[col].trim();
                    filename = filename.replace('\\', File.separatorChar);
                    filename = filename.replace('/', File.separatorChar);
                    if ((filename = filename.substring(filename.lastIndexOf(File.separator) + 1)).length() <= 0 || !(file = new Location(parent, filename)).exists()) continue;
                    referencedLSMs.add(file.getAbsolutePath());
                }
            }
        }
        if (referencedLSMs.size() > 0) {
            return referencedLSMs.toArray(new String[0]);
        }
        String[] fileList = parent.list();
        for (int i = 0; i < fileList.length; ++i) {
            if (!ZeissLSMReader.checkSuffix(fileList[i], new String[]{"lsm"}) || fileList[i].startsWith(".")) continue;
            referencedLSMs.add(new Location(parent, fileList[i]).getAbsolutePath());
        }
        return referencedLSMs.toArray(new String[0]);
    }

    private static Hashtable<Integer, String> createKeys() {
        Hashtable<Integer, String> h = new Hashtable<Integer, String>();
        h.put(new Integer(0x10000001), "Name");
        h.put(new Integer(0x4000000C), "Name");
        h.put(new Integer(0x50000001), "Name");
        h.put(new Integer(-1879048191), "Name");
        h.put(new Integer(-1879048187), "Detection Channel Name");
        h.put(new Integer(-1342177277), "Name");
        h.put(new Integer(-805306367), "Name");
        h.put(new Integer(0x12000001), "Name");
        h.put(new Integer(0x14000001), "Name");
        h.put(new Integer(0x10000002), "Description");
        h.put(new Integer(335544322), "Description");
        h.put(new Integer(0x10000003), "Notes");
        h.put(new Integer(0x10000004), "Objective");
        h.put(new Integer(0x10000005), "Processing Summary");
        h.put(new Integer(0x10000006), "Special Scan Mode");
        h.put(new Integer(0x10000007), "Scan Type");
        h.put(new Integer(0x10000008), "Scan Mode");
        h.put(new Integer(0x10000009), "Number of Stacks");
        h.put(new Integer(0x1000000A), "Lines Per Plane");
        h.put(new Integer(0x1000000B), "Samples Per Line");
        h.put(new Integer(0x1000000C), "Planes Per Volume");
        h.put(new Integer(0x1000000D), "Images Width");
        h.put(new Integer(0x1000000E), "Images Height");
        h.put(new Integer(0x1000000F), "Number of Planes");
        h.put(new Integer(0x10000010), "Number of Stacks");
        h.put(new Integer(0x10000011), "Number of Channels");
        h.put(new Integer(0x10000012), "Linescan XY Size");
        h.put(new Integer(0x10000013), "Scan Direction");
        h.put(new Integer(0x10000014), "Time Series");
        h.put(new Integer(0x10000015), "Original Scan Data");
        h.put(new Integer(0x10000016), "Zoom X");
        h.put(new Integer(0x10000017), "Zoom Y");
        h.put(new Integer(0x10000018), "Zoom Z");
        h.put(new Integer(0x10000019), "Sample 0X");
        h.put(new Integer(0x1000001A), "Sample 0Y");
        h.put(new Integer(0x1000001B), "Sample 0Z");
        h.put(new Integer(0x1000001C), "Sample Spacing");
        h.put(new Integer(0x1000001D), "Line Spacing");
        h.put(new Integer(0x1000001E), "Plane Spacing");
        h.put(new Integer(0x1000001F), "Plane Width");
        h.put(new Integer(0x10000020), "Plane Height");
        h.put(new Integer(0x10000021), "Volume Depth");
        h.put(new Integer(268435508), "Rotation");
        h.put(new Integer(268435509), "Precession");
        h.put(new Integer(268435510), "Sample 0Time");
        h.put(new Integer(268435511), "Start Scan Trigger In");
        h.put(new Integer(268435512), "Start Scan Trigger Out");
        h.put(new Integer(268435513), "Start Scan Event");
        h.put(new Integer(0x10000040), "Start Scan Time");
        h.put(new Integer(0x10000041), "Stop Scan Trigger In");
        h.put(new Integer(268435522), "Stop Scan Trigger Out");
        h.put(new Integer(268435523), "Stop Scan Event");
        h.put(new Integer(0x10000044), "Stop Scan Time");
        h.put(new Integer(268435525), "Use ROIs");
        h.put(new Integer(268435526), "Use Reduced Memory ROIs");
        h.put(new Integer(268435527), "User");
        h.put(new Integer(268435528), "Use B/C Correction");
        h.put(new Integer(268435529), "Position B/C Contrast 1");
        h.put(new Integer(0x10000050), "Position B/C Contrast 2");
        h.put(new Integer(0x10000051), "Interpolation Y");
        h.put(new Integer(268435538), "Camera Binning");
        h.put(new Integer(268435539), "Camera Supersampling");
        h.put(new Integer(268435540), "Camera Frame Width");
        h.put(new Integer(0x10000055), "Camera Frame Height");
        h.put(new Integer(268435542), "Camera Offset X");
        h.put(new Integer(268435543), "Camera Offset Y");
        h.put(new Integer(0x40000001), "Multiplex Type");
        h.put(new Integer(0x40000002), "Multiplex Order");
        h.put(new Integer(0x40000003), "Sampling Mode");
        h.put(new Integer(0x40000004), "Sampling Method");
        h.put(new Integer(0x40000005), "Sampling Number");
        h.put(new Integer(0x40000006), "Acquire");
        h.put(new Integer(0x50000002), "Acquire");
        h.put(new Integer(0x7000000B), "Acquire");
        h.put(new Integer(-1879048188), "Acquire");
        h.put(new Integer(-805306345), "Acquire");
        h.put(new Integer(0x40000007), "Sample Observation Time");
        h.put(new Integer(0x40000008), "Time Between Stacks");
        h.put(new Integer(0x4000000D), "Collimator 1 Name");
        h.put(new Integer(0x4000000E), "Collimator 1 Position");
        h.put(new Integer(0x4000000F), "Collimator 2 Name");
        h.put(new Integer(0x40000010), "Collimator 2 Position");
        h.put(new Integer(0x40000011), "Is Bleach Track");
        h.put(new Integer(1073741842), "Bleach After Scan Number");
        h.put(new Integer(1073741843), "Bleach Scan Number");
        h.put(new Integer(0x40000014), "Trigger In");
        h.put(new Integer(301989892), "Trigger In");
        h.put(new Integer(335544323), "Trigger In");
        h.put(new Integer(1073741845), "Trigger Out");
        h.put(new Integer(301989893), "Trigger Out");
        h.put(new Integer(0x14000004), "Trigger Out");
        h.put(new Integer(1073741846), "Is Ratio Track");
        h.put(new Integer(1073741847), "Bleach Count");
        h.put(new Integer(1073741848), "SPI Center Wavelength");
        h.put(new Integer(1073741849), "Pixel Time");
        h.put(new Integer(0x40000020), "ID Condensor Frontlens");
        h.put(new Integer(1073741857), "Condensor Frontlens");
        h.put(new Integer(0x40000022), "ID Field Stop");
        h.put(new Integer(1073741859), "Field Stop Value");
        h.put(new Integer(0x40000024), "ID Condensor Aperture");
        h.put(new Integer(1073741861), "Condensor Aperture");
        h.put(new Integer(1073741862), "ID Condensor Revolver");
        h.put(new Integer(1073741863), "Condensor Revolver");
        h.put(new Integer(1073741864), "ID Transmission Filter 1");
        h.put(new Integer(1073741865), "ID Transmission 1");
        h.put(new Integer(0x40000030), "ID Transmission Filter 2");
        h.put(new Integer(1073741873), "ID Transmission 2");
        h.put(new Integer(1073741874), "Repeat Bleach");
        h.put(new Integer(0x40000033), "Enable Spot Bleach Pos");
        h.put(new Integer(0x40000034), "Spot Bleach Position X");
        h.put(new Integer(1073741877), "Spot Bleach Position Y");
        h.put(new Integer(1073741878), "Bleach Position Z");
        h.put(new Integer(0x50000003), "Power");
        h.put(new Integer(-1879048190), "Power");
        h.put(new Integer(0x70000003), "Detector Gain");
        h.put(new Integer(0x70000005), "Amplifier Gain");
        h.put(new Integer(0x70000007), "Amplifier Offset");
        h.put(new Integer(0x70000009), "Pinhole Diameter");
        h.put(new Integer(0x7000000C), "Detector Name");
        h.put(new Integer(0x7000000D), "Amplifier Name");
        h.put(new Integer(0x7000000E), "Pinhole Name");
        h.put(new Integer(0x7000000F), "Filter Set Name");
        h.put(new Integer(0x70000010), "Filter Name");
        h.put(new Integer(1879048211), "Integrator Name");
        h.put(new Integer(1879048212), "Detection Channel Name");
        h.put(new Integer(1879048213), "Detector Gain B/C 1");
        h.put(new Integer(1879048214), "Detector Gain B/C 2");
        h.put(new Integer(0x70000017), "Amplifier Gain B/C 1");
        h.put(new Integer(1879048216), "Amplifier Gain B/C 2");
        h.put(new Integer(1879048217), "Amplifier Offset B/C 1");
        h.put(new Integer(0x70000020), "Amplifier Offset B/C 2");
        h.put(new Integer(1879048225), "Spectral Scan Channels");
        h.put(new Integer(0x70000022), "SPI Wavelength Start");
        h.put(new Integer(1879048227), "SPI Wavelength End");
        h.put(new Integer(1879048230), "Dye Name");
        h.put(new Integer(-805306348), "Dye Name");
        h.put(new Integer(0x70000027), "Dye Folder");
        h.put(new Integer(-805306347), "Dye Folder");
        h.put(new Integer(-1879048189), "Wavelength");
        h.put(new Integer(-1879048186), "Power B/C 1");
        h.put(new Integer(-1879048185), "Power B/C 2");
        h.put(new Integer(-1342177279), "Filter Set");
        h.put(new Integer(-1342177278), "Filter");
        h.put(new Integer(-805306364), "Color");
        h.put(new Integer(-805306363), "Sample Type");
        h.put(new Integer(-805306362), "Bits Per Sample");
        h.put(new Integer(-805306361), "Ratio Type");
        h.put(new Integer(-805306360), "Ratio Track 1");
        h.put(new Integer(-805306359), "Ratio Track 2");
        h.put(new Integer(-805306358), "Ratio Channel 1");
        h.put(new Integer(-805306357), "Ratio Channel 2");
        h.put(new Integer(-805306356), "Ratio Const. 1");
        h.put(new Integer(-805306355), "Ratio Const. 2");
        h.put(new Integer(-805306354), "Ratio Const. 3");
        h.put(new Integer(-805306353), "Ratio Const. 4");
        h.put(new Integer(-805306352), "Ratio Const. 5");
        h.put(new Integer(-805306351), "Ratio Const. 6");
        h.put(new Integer(-805306350), "Ratio First Images 1");
        h.put(new Integer(-805306349), "Ratio First Images 2");
        h.put(new Integer(-805306346), "Spectrum");
        h.put(new Integer(301989891), "Interval");
        return h;
    }

    private Integer readEntry() throws IOException {
        return new Integer(this.in.readInt());
    }

    private Object readValue() throws IOException {
        int blockType = this.in.readInt();
        int dataSize = this.in.readInt();
        switch (blockType) {
            case 4: {
                return new Long(this.in.readInt());
            }
            case 5: {
                return new Double(this.in.readDouble());
            }
            case 2: {
                String s = this.in.readString(dataSize).trim();
                StringBuffer sb = new StringBuffer();
                for (int i = 0; i < s.length() && s.charAt(i) >= '\n'; ++i) {
                    sb.append(s.charAt(i));
                }
                return sb.toString();
            }
            case 0: {
                return null;
            }
        }
        this.in.skipBytes(dataSize);
        return "";
    }

    class Marker
    extends SubBlock {
        Marker() {
        }
    }

    class Timer
    extends SubBlock {
        Timer() {
        }
    }

    class BeamSplitter
    extends SubBlock {
        public String filter;
        public String filterSet;

        BeamSplitter() {
        }

        protected void read() throws IOException {
            super.read();
            this.filter = this.getStringValue(-1342177278);
            if (this.filter != null) {
                this.filter = this.filter.trim();
                if (this.filter.length() == 0 || this.filter.equals("None")) {
                    this.filter = null;
                }
            }
            this.filterSet = this.getStringValue(-1342177277);
        }
    }

    class DataChannel
    extends SubBlock {
        public String name;

        DataChannel() {
        }

        protected void read() throws IOException {
            super.read();
            this.name = this.getStringValue(-805306367);
            for (int i = 0; i < this.name.length(); ++i) {
                if (this.name.charAt(i) >= '\n') continue;
                this.name = this.name.substring(0, i);
                break;
            }
            this.acquire = this.getIntValue(-805306345) != 0;
        }
    }

    class IlluminationChannel
    extends SubBlock {
        public Integer wavelength;
        public Float attenuation;

        IlluminationChannel() {
        }

        protected void read() throws IOException {
            super.read();
            this.wavelength = new Integer(this.getIntValue(-1879048189));
            this.attenuation = new Float(this.getFloatValue(-1879048190));
            this.acquire = this.getIntValue(-1879048188) != 0;
        }
    }

    class DetectionChannel
    extends SubBlock {
        public Float pinhole;
        public Float gain;
        public Float amplificationGain;
        public String filter;
        public String filterSet;
        public String channelName;

        DetectionChannel() {
        }

        protected void read() throws IOException {
            super.read();
            this.pinhole = new Float(this.getFloatValue(0x70000009));
            this.gain = new Float(this.getFloatValue(0x70000003));
            this.amplificationGain = new Float(this.getFloatValue(0x70000005));
            this.filter = this.getStringValue(0x70000010);
            if (this.filter != null) {
                this.filter = this.filter.trim();
                if (this.filter.length() == 0 || this.filter.equals("None")) {
                    this.filter = null;
                }
            }
            this.filterSet = this.getStringValue(0x7000000F);
            this.channelName = this.getStringValue(1879048212);
            this.acquire = this.getIntValue(0x7000000B) != 0;
        }
    }

    class Track
    extends SubBlock {
        public Float timeIncrement;

        Track() {
        }

        protected void read() throws IOException {
            super.read();
            this.timeIncrement = Float.valueOf(this.getFloatValue(0x4000000B));
            this.acquire = this.getIntValue(0x40000006) != 0;
        }
    }

    class Laser
    extends SubBlock {
        public String medium;
        public String type;
        public Double power;

        Laser() {
        }

        protected void read() throws IOException {
            super.read();
            this.type = this.getStringValue(0x50000001);
            if (this.type == null) {
                this.type = "";
            }
            this.medium = "";
            if (this.type.startsWith("HeNe")) {
                this.medium = "HeNe";
                this.type = "Gas";
            } else if (this.type.startsWith("Argon")) {
                this.medium = "Ar";
                this.type = "Gas";
            } else if (this.type.equals("Titanium:Sapphire") || this.type.equals("Mai Tai")) {
                this.medium = "TiSapphire";
                this.type = "SolidState";
            } else if (this.type.equals("YAG")) {
                this.medium = "";
                this.type = "SolidState";
            } else if (this.type.equals("Ar/Kr")) {
                this.medium = "";
                this.type = "Gas";
            }
            this.acquire = this.getIntValue(0x50000002) != 0;
            this.power = this.getDoubleValue(0x50000003);
        }
    }

    class Recording
    extends SubBlock {
        public String description;
        public String name;
        public String binning;
        public String startTime;
        public String correction;
        public String immersion;
        public Integer magnification;
        public Float lensNA;
        public Boolean iris;

        Recording() {
        }

        protected void read() throws IOException {
            int next;
            long stamp;
            super.read();
            this.description = this.getStringValue(0x10000002);
            this.name = this.getStringValue(0x10000001);
            this.binning = this.getStringValue(268435538);
            if (this.binning != null && this.binning.indexOf("x") == -1) {
                this.binning = this.binning.equals("0") ? null : this.binning + "x" + this.binning;
            }
            if ((stamp = (long)(this.getDoubleValue(268435510) * 8.64E7)) > 0L) {
                this.startTime = DateTools.convertDate(stamp, 2);
            }
            ZeissLSMReader.this.zoom = (float)this.getDoubleValue(0x10000016);
            String objective = this.getStringValue(0x10000004);
            this.correction = "";
            if (objective == null) {
                objective = "";
            }
            String[] tokens = objective.split(" ");
            for (next = 0; next < tokens.length && tokens[next].indexOf("/") == -1; ++next) {
                this.correction = this.correction + tokens[next];
            }
            if (next < tokens.length) {
                String p = tokens[next++];
                try {
                    this.magnification = new Integer(p.substring(0, p.indexOf("/") - 1));
                }
                catch (NumberFormatException e) {
                    // empty catch block
                }
                try {
                    this.lensNA = new Float(p.substring(p.indexOf("/") + 1));
                }
                catch (NumberFormatException e) {
                    // empty catch block
                }
            }
            this.immersion = next < tokens.length ? tokens[next++] : "Unknown";
            this.iris = Boolean.FALSE;
            if (next < tokens.length) {
                this.iris = new Boolean(tokens[next++].trim().equalsIgnoreCase("iris"));
            }
        }
    }

    class SubBlock {
        public Hashtable<Integer, Object> blockData;
        public boolean acquire = true;

        public SubBlock() {
            try {
                this.read();
            }
            catch (IOException e) {
                ZeissLSMReader.this.traceDebug(e);
            }
        }

        protected int getIntValue(int key) {
            Object o = this.blockData.get(new Integer(key));
            if (o == null) {
                return -1;
            }
            return !(o instanceof Number) ? -1 : ((Number)o).intValue();
        }

        protected float getFloatValue(int key) {
            Object o = this.blockData.get(new Integer(key));
            if (o == null) {
                return -1.0f;
            }
            return !(o instanceof Number) ? -1.0f : ((Number)o).floatValue();
        }

        protected double getDoubleValue(int key) {
            Object o = this.blockData.get(new Integer(key));
            if (o == null) {
                return -1.0;
            }
            return !(o instanceof Number) ? -1.0 : ((Number)o).doubleValue();
        }

        protected String getStringValue(int key) {
            Object o = this.blockData.get(new Integer(key));
            return o == null ? null : o.toString();
        }

        protected void read() throws IOException {
            this.blockData = new Hashtable();
            Integer entry = ZeissLSMReader.this.readEntry();
            Object value = ZeissLSMReader.this.readValue();
            while (value != null) {
                if (!this.blockData.containsKey(entry)) {
                    this.blockData.put(entry, value);
                }
                entry = ZeissLSMReader.this.readEntry();
                value = ZeissLSMReader.this.readValue();
            }
        }

        public void addToHashtable() {
            Integer[] keys;
            String prefix = this.getClass().getSimpleName() + " #";
            int index = 1;
            while (ZeissLSMReader.this.getSeriesMeta(prefix + index + " Acquire") != null) {
                ++index;
            }
            prefix = prefix + index;
            for (Integer key : keys = this.blockData.keySet().toArray(new Integer[0])) {
                if (metadataKeys.get(key) == null) continue;
                ZeissLSMReader.this.addSeriesMeta(prefix + " " + (String)metadataKeys.get(key), this.blockData.get(key));
            }
            ZeissLSMReader.this.addGlobalMeta(prefix + " Acquire", new Boolean(this.acquire));
        }
    }
}

