function m = leaf_mgen_color( m, varargin )
%m = leaf_mgen_color( m, ... )
%   Associate colours with morphogens.
%   Options:
%       'morphogen':   The morphogens to attach colours to.  This can be a
%                      single morphogen name, an array of morphogen
%                      indexes, or a cell array or morphogen names.
%       'color':       Either a single letter, as used in various Matlab
%                      plotting functions, a single RGB triple (row or
%                      column), or a 3*N array of colours, one for each of
%                      the N morphogens specified by the morphogen
%                      argument.  If you give an N*3 array instead it will
%                      transpose it for you, unless N=3, in which case you
%                      have to get it right yourself.  The colours are the
%                      columns, not the rows.
%                      Instead of a single letter, it can be a pair of
%                      letters, in which case the first is used for the
%                      color of positive values and the second for negative
%                      values.  If numerical values are supplied, color can
%                      be an A*B*2 array, in which color(:,:,1) is used as
%                      described above for the positive values, and
%                      color(:,:,2) is used for the negative values.
%                      If colours are not provided for negative values,
%                      colours opposite to those supplied for positive
%                      values will automatically be used.
%
%   Topics: Morphogens, Plotting.

    if isempty(m), return; end
    [s,ok] = safemakestruct( mfilename(), varargin );
    if ~ok, return; end
    s = defaultfields( s, 'morphogen', [], 'color', 'r' );
    ok = checkcommandargs( mfilename(), s, 'exact', ...
        'morphogen', 'color' );
    if ~ok, return; end
    
    if isempty( s.morphogen )
        return;
    end

    if ischar(s.color)
        if length(s.color) > 2
            complain( '%s: Invalid color name ''%s''.\n', mfilename(), s.color );
            return;
        end
        positiveColor = rgbFromColorName( s.color(1) );
        if length(s.color)==2
            negativeColor = rgbFromColorName( s.color(2) );
        else
            negativeColor = oppositeColor( positiveColor );
        end
        s.color = positiveColor;
        s.color(:,:,2) = negativeColor;
    end
    csz = size(s.color);
    if length(csz) > 3
        complain;
        return;
    end
    if (csz(1) ~= 3) && (csz(2)==3)
        s.color = permute( s.color, [2 1 3:length(size(s.color))] );
    end
    if (size(s.color,2)==1) && (length(s.morphogen) > 1)
        s.color = repmat( s.color, 1, length(s.morphogen) );
    end
    if size(s.color,2) ~= length(s.morphogen)
        complain( '%s: Wrong number of colours: %d found, %d morphogens.\n', ...
            mfilename(), size(s.color,2), length(s.morphogen) );
        return;
    end
    if size(s.color,3)==1
        s.color(:,:,2) = oppositeColor( s.color' )';
    end
    
    mgenindexes = FindMorphogenIndex( m, s.morphogen, mfilename() );
    m.mgenposcolors(:,mgenindexes) = s.color(:,1:length(mgenindexes),1);
    m.mgennegcolors(:,mgenindexes) = s.color(:,1:length(mgenindexes),2);
end

function rgb = rgbFromColorName( c )
    switch c
        case 'r'
            rgb = [1;0;0];
        case 'g'
            rgb = [0;1;0];
        case 'b'
            rgb = [0;0;1];
        case 'c'
            rgb = [0;1;1];
        case 'm'
            rgb = [1;0;1];
        case 'y'
            rgb = [1;1;0];
        case 'o'
            rgb = [1;0.5;0];
        case 'k'
            rgb = [0;0;0];
        case 'w'
            rgb = [1;1;1];
        otherwise
            % Unexpected colour name.
            rgb = [1;1;1];
    end
end
