function ok = checkclonesvalid( secondlayer )
%ok = checkclonesvalid( secondlayer )
%   Make consistency checks of secondlayer.

    ok = 1;
    numcells = length( secondlayer.cells );
    numedges = size( secondlayer.edges, 1 );
    numvxs = length( secondlayer.vxFEMcell );

    for ci=1:numcells
        nv = length( secondlayer.cells(ci).vxs );
        ne = length( secondlayer.cells(ci).edges );
        
        % Each cell must have the same number of vertexes as edges.
        if nv ~= ne
            fprintf( 1, 'Cell %d has %d vertexes and %d edges. Numbers should be equal.\n', ...
                ci, nv, ne );
            ok = 0;
        end
        
        % Each cell must have at least three vertexes.
        if (ne < 3) || (nv < 3)
            fprintf( 1, 'Cell %d has %d vertexes and %d edges. A cell must have at least three of each.\n', ...
                ci, nv, ne );
        end
    end
    
    
    % All indexes must be in range.
    for ci=1:numcells
        bad = find( secondlayer.cells(ci).vxs > numvxs );
        if bad
            fprintf( 1, 'Cell %d has invalid vertex indices', ci );
            fprintf( 1, ' %d', bad );
            fprintf( 1, ': [' );
            fprintf( 1, ' %d', secondlayer.cells(ci).vxs );
            fprintf( 1, ' ]\n' );
            ok = 0;
        end
        bad = find( secondlayer.cells(ci).edges > numedges );
        if bad
            fprintf( 1, 'Cell %d has invalid edge indices', ci );
            fprintf( 1, ' %d', bad );
            fprintf( 1, ': [' );
            fprintf( 1, ' %d', secondlayer.cells(ci).edges );
            fprintf( 1, ' ]\n' );
            ok = 0;
        end
    end
    bad = find( secondlayer.edges(:,1) > numvxs );
    if bad
        fprintf( 1, 'Edges with invalid first vertex index:\n' );
        fprintf( 1, '    edge %d bad vertex %d\n', [ bad; secondlayer.edges(bad,1) ] );
        ok = 0;
    end
    bad = find( secondlayer.edges(:,2) > numvxs );
    if bad
        fprintf( 1, 'Edges with invalid second vertex index:\n' );
        fprintf( 1, '    edge %d bad vertex %d\n', [ bad; secondlayer.edges(bad,2) ] );
        ok = 0;
    end
    bad = find( secondlayer.edges(:,3) > numcells );
    if bad
        fprintf( 1, 'Edges with invalid first cell index:\n' );
        fprintf( 1, '    edge %d bad vertex %d\n', [ bad; secondlayer.edges(bad,3) ] );
        ok = 0;
    end
    bad = find( secondlayer.edges(:,4) > numcells );
    if bad
        fprintf( 1, 'Edges with invalid second cell index:\n' );
        fprintf( 1, '    edge %d bad vertex %d\n', [ bad; secondlayer.edges(bad,4) ] );
        ok = 0;
    end
        
    
    % Each edge of each cell joins the corresponding vertexes of that cell,
    % and the edge data references the cell.
    for ci=1:numcells
        nv = length( secondlayer.cells(ci).vxs );
        cellvxends = secondlayer.cells(ci).vxs([(1:nv)',[2:nv,1]']);
        flips = cellvxends(:,1) > cellvxends(:,2);
        cellvxends(flips,:) = cellvxends(flips,[2 1]);
      % cellvxends = sort( secondlayer.cells(ci).vxs([(1:nv)',[2:nv,1]']), 2 );
        celledgeends = secondlayer.edges( secondlayer.cells(ci).edges, [1 2] );
        flips = celledgeends(:,1) > celledgeends(:,2);
        celledgeends(flips,:) = celledgeends(flips,[2 1]);
        badEdgeVxs = cellvxends ~= celledgeends;
        if any( badEdgeVxs(:) )
            for cvi=1:nv
                ei = secondlayer.cells(ci).edges(cvi);
                e = secondlayer.edges(ei,:);
                if badEdgeVxs(cvi,1) || badEdgeVxs(cvi,2)
                    vi = secondlayer.cells(ci).vxs(cvi);
                    cvi1 = mod(cvi,nv) + 1;
                    vi1 = secondlayer.cells(ci).vxs(cvi1);
                    fprintf( 1, 'Cell %d edge %d (%d) should join vertexes %d and %d, but joins %d and %d. Edge = [ %d %d %d %d ]\n', ...
                        ci, cvi, ei, vi, vi1, e([1 2]), e );
                    ok = 0;
                end
                if (e(3) ~= ci) && (e(4) ~= ci)
                    fprintf( 1, 'Cell %d edge %d (%d) should have cell %d on one side, but has cells %d and %d.\n', ...
                        ci, cvi, ei, ci, e([3 4]) );
                    ok = 0;
                end
            end
        end
    end
    
    tweaked = false;
    % Every edge is an edge of at least one cell.
    lostEdges = true(1,numedges);
    for ci=1:numcells
        lostEdges( secondlayer.cells(ci).edges ) = false;
    end
    if any(lostEdges)
        fprintf( 1, 'Edges' );
        fprintf( 1, ' %d', find(lostEdges) );
        fprintf( 1, ' are not referenced by any cell.\n' );
      % ok = false;
        newToOldEdge = find(~lostEdges);
        oldToNewEdge = zeros(1,numedges);
        oldToNewEdge(newToOldEdge) = 1:length(newToOldEdge);
        for ci=1:numcells
            secondlayer.cells(ci).edges = oldToNewEdge( secondlayer.cells(ci).edges );
        end
        secondlayer.edges = secondlayer.edges(~lostEdges,:);
        numedges = size( secondlayer.edges, 1 );
        tweaked = true;
    end
    
    % Every vertex is a vertex of at least one cell.
    lostVxs = true(1,numvxs);
    for ci=1:numcells
        lostVxs( secondlayer.cells(ci).vxs ) = false;
    end
    if any(lostVxs)
        fprintf( 1, 'Vertexes' );
        fprintf( 1, ' %d', find(lostVxs) );
        fprintf( 1, ' are not referenced by any cell.\n' );
      % ok = false;
        newToOldVx = find(~lostVxs);
        oldToNewVx = zeros(1,numvxs);
        oldToNewVx(newToOldVx) = 1:length(newToOldVx);
        for ci=1:numcells
            secondlayer.cells(ci).vxs = oldToNewVx( secondlayer.cells(ci).vxs );
        end
        secondlayer.vxFEMcell = secondlayer.vxFEMcell( ~lostVxs );
        secondlayer.vxBaryCoords = secondlayer.vxBaryCoords( ~lostVxs, : );
        secondlayer.cell3dcoords = secondlayer.cell3dcoords( ~lostVxs, : );
        secondlayer.edges(:,1:2) = oldToNewVx( secondlayer.edges(:,1:2) );
        numvxs = length( secondlayer.vxFEMcell );
        tweaked = true;
    end
    
    % Every vertex is a vertex of at least one edge.
    lostVxs = true(1,numvxs);
    lostVxs( secondlayer.edges(:,[1 2]) ) = false;
  % for ei=1:numedges
  %     lostVxs( secondlayer.edges(ei,[1 2]) ) = false;
  % end
    if any(lostVxs)
        fprintf( 1, 'Vertexes' );
        fprintf( 1, ' %d', find(lostVxs) );
        fprintf( 1, ' are not referenced by any edge.\n' );
        ok = false;
    end
    
    % Every edge had a generation index.
    if isfield( secondlayer, 'generation' )
        if numedges ~= length(secondlayer.generation)
            fprintf( 1, 'There are %d edges but %d generation indexes.\n', ...
                numedges, length(secondlayer.generation) );
            ok = false;
        end
    end
    
    if ok && tweaked
        fprintf( 1, 'About to redo checkclonesvalid.\n' );
        ok = checkclonesvalid( secondlayer );
    end
end
