function m = leaf_setproperty( m, varargin )
%m = leaf_setproperty( m, ... )
%   Set global properties of the leaf.
%   The arguments are a series of name/value pairs.
%
%   The property names that this applies to are:
%       'poisson'       Poisson's ratio.  The normal value is 0.35 and
%                       there is little need to change this.
%       'bulkmodulus'   The bulk modulus.  The normal value is 3000 and
%                       there is little need to change this.
%       'validate'      Whether to validate the mesh after every iteration.
%                       This is for debugging purposes and should normally
%                       be off.
%       'displayedgrowth'    Specifies which morphogen to plot.
%
%       ...and many others I have omitted to document.
%
%   Example:
%       m = leaf_setproperty( m, 'poisson', 0.49 );
%
%   Equivalent GUI operation: several GUI elements are implemented by this
%   command:
%       poisson:        Text box in "Mesh editor" panel.
%       bulkmodulus:    No GUI equivalent.
%       residstrain:    "Strain retention" in "Simulation" panel.
%       validate:       No GUI equivalent.
%       displayedgrowth:  "Displayed m'gen" menu in "Morphogens" panel.
%       ...etc.
%
%   Topics: Misc.

    if isempty(m), return; end
    [s,ok] = safemakestruct( mfilename(), varargin );
    if ~ok, return; end
    if isempty(s), return; end

    dictionary = struct( ...
        'poisson', 'poissonsRatio', ...
        'do_validate', 'validateMesh', ...
        'do_growth', 'growthEnabled', ...
        'do_diffusion', 'diffusionEnabled', ...
        'do_interaction', 'allowInteraction', ...
        'do_splitlongfem', 'allowSplitLongFEM', ...
        'do_splitbentfem', 'allowSplitBentFEM', ...
        'do_splitbio', 'allowSplitBio', ...
        'do_flip', 'allowFlipEdges', ...
        'usetensors', 'useGrowthTensors', ...
        'plastic', 'plasticGrowth' ...
    );
    
    staticchanged = false;
    for i=1:2:(length(varargin)-1)
        fieldname = varargin{i};
        if isfield( dictionary, fieldname )
            fieldname = dictionary.(varargin{i});
        end
        if isfield( m.globalProps, fieldname )
            m.globalProps.(fieldname) = varargin{i+1};
            staticchanged = true;
        elseif isfield( m.globalDynamicProps, fieldname )
            m.globalDynamicProps.(fieldname) = varargin{i+1};
        elseif ischar( varargin{i} )
            fprintf( 1, '%s: unexpected property ''%s'' ignored.\n', mfilename(), varargin{i} );
        else
            fprintf( 1, '%s: property name expected, object of type "%s" found.\n', ...
                mfilename(), class( varargin{i} ) );
        end
    end
    
    if isfield(s,'poisson') || isfield(s,'poissonsRatio') || isfield(s,'bulkmodulus')
        m = updateElasticity( m );
    end
    
    if isfield(s,'colors') || isfield(s,'colorvariation')
        m.globalProps.colorparams = ...
            makesecondlayercolorparams( m.globalProps.colors, m.globalProps.colorvariation );
    end
    
    if staticchanged
        saveStaticPart( m );
    end

    m.saved = 0;
end

