function varargout = getSurfaceData(varargin)
% function varargout = getSurfaceData(varargin)
%
% Description:
%
% Returns the surface data for visualisation
%
% Author: Johann Strasser
% Date: 080130

% Initialise surface data structure
surfaceData.templatePoints = [];
surfaceData.templateSurface = [];
surfaceData.handle = [];
surfaceData.lightHandles = []; % We'll need more than one light for decent visualisation, two should suffice
surfaceData.colour = [0.5, 0.5, 0.5];
surfaceData.templateVoxelSize = [];
surfaceData.templateVolumeSize = [];
surfaceData.currentModelSurface.vertices = [];
surfaceData.currentModelSurface.faces = [];
surfaceData.currentModelSurface.normals = [];

% Declare variables
name = [];
mode = 'template';
surfaceDirectory = [];
controlPointsDirectory = [];

for i=1:2:length(varargin)
    switch lower(varargin{i})
        case 'name'
            name = varargin{i+1};
        case 'mode'
            mode = varargin{i+1};
        case 'surfacedirectory'
            surfaceDirectory = varargin{i+1};
        case 'controlpointsdirectory'
            controlPointsDirectory = varargin{i+1};
        otherwise
            error(['Unknown parameter name passed to ', mfilename, '.  Name was ' varargin{i}])
    end
end

if isempty(name)
    error([mfilename, ': Parameter ''name'' not specified or empty.']);
end

if isempty(surfaceDirectory)
    error([mfilename, ': Parameter ''surfaceDirectory'' not specified or empty.']);
end

if isempty(controlPointsDirectory) && ~isequal(mode, 'template')
    error([mfilename, ': Parameter ''controlPointsDirectory'' not specified or empty.']);
else
    controlPointsDirectory = surfaceDirectory;
end

% Get the voxel size
surfaceData.templateVoxelSize = getVoxelSizeFromString('String', name);

fullSurfaceName = fullfile(surfaceDirectory, [name, '.isosurf']);

if exist(fullSurfaceName, 'file')
    load(fullSurfaceName, 'isosurf', '-mat');
    surfaceData.templateSurface = isosurf;

    if isequal(mode, 'template')
        % Get the template points
        fullTemplateFileName = fullfile(controlPointsDirectory, ...
            [name, '.temp_dat']);

        load(fullTemplateFileName, 'PMTemplate', '-mat');
        surfaceData.templatePoints = get(PMTemplate, 'pts')';

        % Get the volume size from the template extensions
        fullTemplateExtensionsName = fullfile(controlPointsDirectory, ...
            [name, '.temp_ex_dat']);

        load(fullTemplateExtensionsName, 'pmtExtensions', '-mat');
        surfaceData.templateVolumeSize = pmtExtensions.volumeSize;

        % Note that the surface points are in volume space, but the template points
        % are in real space, which has its origin in the centre of the volume and
        % is scaled by the voxel size. Hence we have to align the space of the
        % template point set and the template surface. We only need to do this once

        % Shift the origin of the surface space to the middle of the volume
        % Need to know the volume dimensions
        surfaceData.templateSurface.vertices = surfaceData.templateSurface.vertices - ...
            repmat(surfaceData.templateVolumeSize(1:3) , ...
            size(surfaceData.templateSurface.vertices, 1), 1) / 2;

        % Scale the surface space by the voxel size
        % Need to know the voxel size (in the template name)
        scaleMat = diag(surfaceData.templateVoxelSize);
        surfaceData.templateSurface.vertices = (scaleMat * surfaceData.templateSurface.vertices')';

        % Now swap x and y dimensions since we are going from matlab volume space
        % to real space
        x = surfaceData.templateSurface.vertices(:, 1);
        surfaceData.templateSurface.vertices(:, 1) = surfaceData.templateSurface.vertices(:, 2);
        surfaceData.templateSurface.vertices(:, 2) = x;

        % Initialise java implementation of general MCT
        % mctroot = fileparts(which('mctj_demo.m'));
        % mctj_initialize_java('mctroot', mctroot);
    end
end

varargout{1} = surfaceData;
