function varargout = loadVolumeFromSlicesDirTyped(varargin)
% function varargout = loadVolumeFromSlicesDir(varargin)
% 
% Description:
% Function to load slices given the directory where the slices reside and
% the extension of the graphics file type.
% 
% Author: Johann Strasser
% Date: 070312

fullVolumePath = pwd;
extension = 'png';
message = 'Reading volume from disk...';
volume = [];

if length(varargin)>1
    for i = 1:2:length(varargin)
        switch(lower(varargin{i}))
            case 'fullvolumepath'
                fullVolumePath = varargin{i + 1};
            case 'extension'
                extension = varargin{i + 1};
            case 'message'
                message = varargin{i + 1};
            otherwise
                error(['Unknown parameter name passed to ', mfilename, '.  Name was ' varargin{i}]);
        end   
  end
end

files = dir(fullfile(fullVolumePath, ['*', extension]));

% Sort the filenames in ASCII dictionary order, just in case the list
% returned by the operating system is ordered differently
fileNames = sort({files.name});
noOfSlices = length(fileNames);

% Check if there is a reconstruction reference file which is used as a 
% thumbnail by the reconstruction software and does not belong to the volume.
% The filename of such a file has a constant part we can look for, e.g if a
% slice is called 999_rec0510.bmp then the thumbnail file is called 
% 999_rec_spr.bmp. Hence we look for the _rec_spr part and eliminate the
% file from the list if found.
isRecFile = zeros(1, noOfSlices);

for i = 1:noOfSlices
    isFound = findstr('_rec_spr', fileNames{i});

    if ~isempty(isFound)
        isRecFile(i) = 1;
        disp(sprintf('%s:  Reconstruction software thumbnail found. File will be ignored.', mfilename));
    end
end

fileNames(find(isRecFile)) = [];
noOfSlices = length(fileNames);

isSingleChannel = 0;
   
if noOfSlices > 0
    
    % Get image dimensions and number of images to determine required space
    im = imread(fullfile(fullVolumePath, fileNames{1}));
    dims = size(im);
    
    % We may have 8 bit images
    if length(dims) < 3
        isSingleChannel = 1;
        dims = [dims(1:2), noOfSlices];
        noOfElements = prod(dims);
        disp(sprintf('%s:  Volume dimensions: %d x %d x %d', mfilename, dims(1), dims(2), dims(3)));
    else
        % Swap size of third and fourth dimension in order to have the rgb vector
        % along the fourth dimension. Hence we later allocate a m x n x p x 3
        % matrix
        dims = [dims(1:2), noOfSlices, dims(3)];
        noOfElements = prod(dims);
        disp(sprintf('%s:  Volume dimensions: %d x %d x %d x %d', mfilename, dims(1), dims(2), dims(3), dims(4)));
    end

    disp(sprintf('%s:  Volume (uint8) size in MB: %5.2f', mfilename, noOfElements / 2^20));

    % Allocate memory
    volume = zeros(dims, 'uint8');
    h = waitbar(0, message);

    % Load images
    for i = 1:noOfSlices
        im = imread(fullfile(fullVolumePath, fileNames{i}));
        
        if isSingleChannel
            volume(:, :, i) = im(:, :);
        else
            volume(:, :, i, :) = im(:, :, :);
        end
                
        waitbar(i/noOfSlices);
    end
    
    if(ishandle(h))
        close(h);
    end
end

varargout{1} = volume;