/*****************************************************************************

FILE
    sieve2d.c

AUTHOR
    rwy in collaboration with J. Andrew Bangham, UEA

MODIFICATION HISTORY

    1.1     26:jun:95   rwy   First version submitted to SCCS.
    1.2     27:jun:95   rwy   Treat single PicElementPtr regions separately
                              when finding boundary PictureElement & extrema.
    1.3     27:jun:95   rwy   Make alint compliant.
    1.4     27:jun:95   rwy   Boundary PictureElement added differently.
    1.5     27:jun:95   rwy   Expand in-line the key loop when
                              finding boundary PictureElement.
    1.6     04:jul:95   rwy   Improved when merging extrema with
                              much larger regions.
    1.7     05:jul:95   rwy   Faster implementation of v1.6 mods.
    1.8     26:jul:95   rwy   Implement open, close, M, N sieves.
    1.9     27:jul:95   rwy   Implement band-pass option.
    1.10    03:aug:95   rwy   Bug fix and make alint compliant.
    1.11    09:aug:95   rwy   Bug fixes and add granule output.
    1.12    14:sep:95   rwy   Add edge output.
    1.13    15:apr:98   rwy   Converted to Matlab 5.2.  Implemented
                              uint8 inputs and int16 outputs.
                              Improved variable typing.
    1.14    15:apr:98   rwy   Tidied formatting of code.
    1.15    16:apr:98   rwy   Further tidying.
    1.16    16:apr:98   rwy   Output granules using Matlab structure.
    1.17    17:apr:98   rwy   Partitioned code into sub-functions.
                              Improved trapping of list overflows.
    1.18    20:apr:98   rwy   Minor tweaks.
    1.19    20:apr:98   rwy   Changed granule storage to struct of
                              arrays rather than array of structures
                              for reduced memory usage.
    1.20    24:apr:98   rwy   Changed processing order of extrema for
                              median filter to be a closer approximation
                              to the spatial order within the image.

DESCRIPTION

    Program to compute a fast, memory efficient connected-set datasieve.
	1D, 2D and 3D versions exist.

SCCS IDENTIFIER
    @(#)sieve2d.c	1.20 04/24/98

Updates
1998 to 2013 J. Andrew Bangham.

Revision to new working version re-committed
16/10/2013

Current problem: appears to be associated with allocating ...
        *GranuleListBasePtr = (int32 *)mxCalloc((unsigned int)(GRANSIZE*10*NumPictureElements), sizeof(int32));


******************************************************************************/


#include "mex.h"

#define bool    unsigned char
#define uint8   unsigned char
#define int16   short int
#define int32   long int
#define pointer int32

#define FALSE   0
#define TRUE    1

/* Define filter types */
#define MEDIAN    ('m')     /* process max & min in any order */
#define OPENING   ('o')     /* process max only */
#define CLOSING   ('c')     /* process min only */
#define OPENCLOSE ('M')     /* process max than min at each mesh */
#define CLOSEOPEN ('N')     /* process min then max at each mesh */

/* Define output types */
#define LOWPASS   ('l')
#define BANDPASS  ('b')
#define FUSEBPASS ('f')
#define GRANULES  ('g')
#define VERBOSE   ('v')
#define EDGES     ('e')


/****************************************************************************/

/* PARAMETER DEFINITIONS */

/* Maximum length of mesh and filter input arguments */
#define MAXARGS  256

/* Length of hash table used to list adjacent regions */
#define HASHLEN  2048

/* Maximum number of equal amplitude adjacent regions */
#define MAXEQ    1024

/* Define space for granules (x number of NumPictureElements) (Not sure why this particular size) */
#define GRANSIZE 8

/* Define space for extrema (x number of NumPictureElements) */
#define VERBOSESIZE 8


/****************************************************************************/

/* STRUCTURE DEFINITIONS */

/* Information stored for each image PicElementPtr (14 bytes/PicElementPtr) */
typedef struct Region_Graph_Vertex
{
    uint8   value;      /* Pel intensity */
    uint8   AdjacencyFlags;  /* Flags to indicate adjacency */
    pointer root_PicElementPtr;   /* Pointer to root PicElementPtr (or area of region) */
    pointer next_PicElementPtr;   /* Pointer to next PicElementPtr in region */
    pointer OffsetToNextBoundaryPicElementPtr;   /* Pointer to next distinct boundary PicElementPtr */
	double last_area;
} REGION_GRAPH_VERTEX;

/* Information stored for each extremum (8 bytes/extremum = 4 bytes/PicElementPtr) */
typedef struct Extrema_Vertex
{
    pointer root_PicElementPtr;   /* Pointer to root PicElementPtr */
    pointer NextExtremum;   /* Pointer to next extremum in list */
} EXTREMAL_VERTEX;

/* Coordinates of a PicElementPtr */
typedef struct PicElementPtr
{
    int16 m, n;
    bool edge;
} PICTURE_ELEMENT;


/****************************************************************************/

/* FUNCTION PROTOTYPES */

void apply_sieve(uint8 *X, REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, EXTREMAL_VERTEX *Min, EXTREMAL_VERTEX *MaximaList,
             uint8 *BandPassOutput1, uint8 *BandPassOutput2, int32 *GranuleListBasePtr, int32 *GranuleCounter, PICTURE_ELEMENT *PicElementPtr, int32 num_mesh,
             int32 *meshes, int32 *filters, int32 *outputs, int32 out_type,
             int32 M, int32 N, int32 ConnectivityNdir, mxArray **plhs,double ratio, int32 minarea, int32 maxarea, int32 minamp);
void parse_extrema_list(REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, EXTREMAL_VERTEX *StartOfMinimaList, EXTREMAL_VERTEX *StartOfMaximaList,
             int32 *Min_min_area, int32 *Max_min_area, int32 area,
             int32 filter, int32 *offsets, int32 **GranuleListAlsoCurrentPtr, int32 *GranuleCounter, double ratio, int32 minarea, int32 maxarea, int32 minamp);
int32 process_extremum(REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, REGION_GRAPH_VERTEX **RegionGraphBaseReturnedVertexPtr, int32 *offsets,
             int32 **GranuleListAlsoCurrentPtr, int32 *GranuleCounter, double ratio, int32 minarea, int32 maxarea, int32 minamp);
void find_connected_sets(REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, uint8 *X, int32 imRows, int32 imCols,
             int32 ConnectivityNdir);
void find_adjacency_and_extrema(REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, EXTREMAL_VERTEX *StartOfMinimaList,
             EXTREMAL_VERTEX *StartOfMaximaList, int32 *Min_min_area, int32 *Max_min_area,
             int32 filter, int32 imRows, int32 imCols, int32 ConnectivityNdir);
int32 check_lists(bool reset, pointer adjacencyRoot_ptr);
void process_mesh1(uint8 *X, int32 filter, int32 imRows, int32 imCols, int32 ConnectivityNdir);

void copy_image_double(uint8 *Y, double *X, int32 NumPictureElements);
void copy_image_uint8(uint8 *Y, uint8 *X, int32 NumPictureElements);
void duplicate_image_uint8(uint8 *Y, uint8 *X, int32 NumPictureElements);
void make_lowpass_image_int16(int16 *Y, REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, int32 NumPictureElements);
void make_lowpass_image_uint8(uint8 *Y, REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, int32 NumPictureElements);
void make_bandpass_image_int16(int16 *Y, uint8 *X1, uint8 *X2, int32 NumPictureElements);
void make_fused_image_int16(int16 *Y, PICTURE_ELEMENT *PicElementPtr, int32 ConnectivityNdir, int32 imRows,
             int32 imCols, bool edges);

void allocate_memory(uint8 **X, REGION_GRAPH_VERTEX **RegionGraphBaseVertexPtr, EXTREMAL_VERTEX **Min, EXTREMAL_VERTEX **MaximaList,
             uint8 **BandPassOutput1, uint8 **BandPassOutput2, PICTURE_ELEMENT **PicElementPtr, int32 **GranuleListBasePtr, int32 NumPictureElements,
             int32 out_type);
void create_matlab_output(mxArray **mY, int32 out_type, REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, uint8 *BandPassOutput1,
             uint8 *BandPassOutput2, int32 *GranuleListBasePtr, int32 *GranuleListCurrentPtr, int32 *GranuleCounter, 
			 PICTURE_ELEMENT *PicElementPtr, int32 imRows, int32 imCols, int32 ConnectivityNdir);


/****************************************************************************/

/* GATEWAY FUNCTION */

/* Gateway function to interface with Matlab */
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
    const mxArray *mxArrayInputImage, *mM, *mF, *mO;
    uint8 *ByteImagePtr, *BandPassOutput1, *BandPassOutput2;
    int32 num_mesh, meshes[MAXARGS], outputs[MAXARGS], filters[MAXARGS],
          out_type, ConnectivityNdir, *GranuleListBasePtr, GranuleCounter=0, imRows, imCols, k;
    REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr;
    EXTREMAL_VERTEX *MaximaList, *MinimaList;
    PICTURE_ELEMENT *PicElementPtr;
	double *parameters,ratio,levels,minarea,maxarea,minamp,Mrows,Ncols; /* for MSERs */


    /* Check for correct number of arguments */
    if (nrhs<2 || nrhs>7)
        mexErrMsgTxt("Wrong number of input arguments");

    for (k=0; k<nrhs-1; k++)
    {
        if (mxIsComplex(prhs[0]))
            mexErrMsgTxt("Input arguments must be REAL");
    }

    /* Get input image */
    if (!mxIsUint8(prhs[0]) && !mxIsDouble(prhs[0]))
        mexErrMsgTxt("Input image must be DOUBLE or UINT8 array");
    mxArrayInputImage = prhs[0];
    imRows = (int32)mxGetM(mxArrayInputImage);
    imCols = (int32)mxGetN(mxArrayInputImage);
	if (imRows*imCols > 2048*2048)
    //if (imRows > 2048 || imCols > 2048)
        mexErrMsgTxt("Input image exceeds maximum allowed dimensions");

    /* Check mesh specifiers */
    mM = prhs[1];
    if (!mxIsDouble(mM))
        mexErrMsgTxt("Meshes must be DOUBLE vector");
    num_mesh = (int32)(mxGetM(mM)*mxGetN(mM));
    if (num_mesh > MAXARGS)
        mexErrMsgTxt("Meshes vector exceeds maximum allowed length");

    /* Check filter specifiers */
    if (nrhs>=3)
    {
        mF = prhs[2];
        if (!mxIsChar(mF))
            mexErrMsgTxt("Filter types must be CHAR vector");
        if (mxGetM(mF)*mxGetN(mF)!=num_mesh)
            mexErrMsgTxt("Filter types length is invalid");
    }

    /* Check output specifiers */
    if (nrhs>=4)
    {
        mO = prhs[3];
        if (!mxIsDouble(mO))
            mexErrMsgTxt("Output indices must be DOUBLE vector");
        if (mxGetM(mO)*mxGetN(mO)!=num_mesh)
            mexErrMsgTxt("Output indices length is invalid");
    }

    /* Get mesh, filter and output specifiers */
    for (k=0; k<num_mesh; k++)
    {
        meshes[k] = (int32)*(mxGetPr(mM)+k);
        if (meshes[k] < 0)
            mexErrMsgTxt("Meshes must be positive");

        if (nrhs>=3)
        {
            filters[k] = (int32)*((int16 *)mxGetPr(mF)+k);
            if (   filters[k]!=MEDIAN
                && filters[k]!=OPENING   && filters[k]!=CLOSING
                && filters[k]!=OPENCLOSE && filters[k]!=CLOSEOPEN)
                mexErrMsgTxt("Filter type must be 'm','o','c','M' or 'N'");
        }
        else
            filters[k] = MEDIAN;

        if (nrhs>=4)
        {
            outputs[k] = (int32)*(mxGetPr(mO)+k);
            if (outputs[k]<0 || (outputs[k]>nlhs && outputs[k]>1))
                mexErrMsgTxt("Output index is invalid");
        }
        else
            outputs[k] = (k==num_mesh-1) ? 1 : 0;
    }

    /* Get output type */
    if (nrhs>=5)
    {
        if (!mxIsChar(prhs[4]) || mxGetM(prhs[4])*mxGetN(prhs[4])!=1)
            mexErrMsgTxt("Output type must be CHAR scalar");
        out_type = (int32)*(int16 *)mxGetPr(prhs[4]);
        if (   out_type!='l'&& out_type!='b' && out_type!='f'
            && out_type!='e' && out_type!='g' && out_type!='v')
            mexErrMsgTxt("Output type must by 'l','b','f', 'e', 'v' or 'g'");
    }
    else
        out_type = LOWPASS;

    /* Get connectivity */
    if (nrhs==6)
    {
        if (!mxIsDouble(prhs[5]))
            mexErrMsgTxt("Connectivity must be DOUBLE scalar");
        ConnectivityNdir = (int32)mxGetScalar(prhs[5]);
        if (ConnectivityNdir!=4 && ConnectivityNdir!=8)
            mexErrMsgTxt("Connectivity must be 4 or 8");
    }
    else
        ConnectivityNdir = 4;

	/* Get MSER parameters */
    levels=256;
	ratio = 0;
	minarea=0;
	maxarea=100;
	minamp=0;
    if (nrhs==7)
    {
        if (!mxIsDouble(prhs[6]))
            mexErrMsgTxt("Extrema parameters must be DOUBLE scalar (up to 5 element vector)");
		Mrows=(double)mxGetM(prhs[6]);
		Ncols=(double)mxGetN(prhs[6]);
		if (Mrows*Ncols>=1)
		{
	        parameters =mxGetPr(prhs[6]);
			levels=parameters[0];
			if (Mrows*Ncols>=2)
				ratio=parameters[1];
			if (Mrows*Ncols>=3)
				minarea=parameters[2];
			if (Mrows*Ncols>=4)
				maxarea=parameters[3];
			if (Mrows*Ncols>=5)
				minamp=parameters[4];
		}
        if (levels<0 || levels>256)
            mexErrMsgTxt("levels must be in range 2=<levels<256");
        if (ratio<0)
            mexErrMsgTxt("ratio must be in range 0=<ratio");
        if (minarea<0 )
            mexErrMsgTxt("minarea must be in range 0<minarea");
    }
    /* Allocate dynamic memory */
    allocate_memory(&ByteImagePtr, &RegionGraphBaseVertexPtr, &MinimaList, &MaximaList, 
		&BandPassOutput1, &BandPassOutput2, &PicElementPtr, &GranuleListBasePtr, imRows*imCols, out_type);


    /* Copy input image to uint8 array */
    if (mxIsDouble(mxArrayInputImage))
        copy_image_double(ByteImagePtr, (double *)mxGetPr(mxArrayInputImage), imRows*imCols);
    else
        copy_image_uint8(ByteImagePtr, (uint8 *)mxGetPr(mxArrayInputImage), imRows*imCols);

    /* Apply sieve */
    apply_sieve(ByteImagePtr, RegionGraphBaseVertexPtr, MinimaList, MaximaList, BandPassOutput1, 
		BandPassOutput2, GranuleListBasePtr, &GranuleCounter,  PicElementPtr, num_mesh, meshes, filters,
                outputs, out_type, imRows, imCols, ConnectivityNdir, plhs, ratio, minarea, maxarea, minamp);
}


/****************************************************************************/

/* APPLY SIEVE */

void apply_sieve(uint8 *ByteImagePtr, REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, EXTREMAL_VERTEX *MinimaList, EXTREMAL_VERTEX *MaximaList,
                 uint8 *BandPassOutput1, uint8 *BandPassOutput2, int32 *GranuleListBasePtr, int32 *GranuleCounter, 
				 PICTURE_ELEMENT *PicElementPtr, int32 num_mesh,
                 int32 *meshes, int32 *filters, int32 *outputs,
                 int32 out_type, int32 imRows, int32 imCols, int32 ConnectivityNdir, mxArray **plhs, double ratio, int32 minarea, int32 maxarea, int32 minamp)
{
    int32 max_mesh, Min_min_area, Max_min_area, output, filter, *GranuleListCurrentPtr,
          offsets[8], min_area, k;
    uint8 *Btmp;
    bool done;

    /* Set offsets in each of the 8 directions ( Matlab stores arrays column first )*/
    offsets[0]=-1;   offsets[1]=+1;   offsets[2]=-imRows;   offsets[3]=+imRows;
    offsets[4]=-1-imRows; offsets[5]=-1+imRows; offsets[6]=+1-imRows; offsets[7]=+1+imRows;

    /* Duplicate original image for bandpass outputs */
    if (BandPassOutput1!=0)
        duplicate_image_uint8(BandPassOutput1, ByteImagePtr, imRows*imCols);

    /* Construct connected set graph and extrema list */
    if ((out_type!=GRANULES || outputs[0]==0) && meshes[0]!=0)
    {
        /* Process mesh 1 */
        if (filters[0]==OPENCLOSE)
        {
            process_mesh1(ByteImagePtr, OPENING, imRows, imCols, ConnectivityNdir);
            process_mesh1(ByteImagePtr, CLOSING, imRows, imCols, ConnectivityNdir);
        }
        else if (filters[0]==CLOSEOPEN)
        {
            process_mesh1(ByteImagePtr, CLOSING, imRows, imCols, ConnectivityNdir);
            process_mesh1(ByteImagePtr, OPENING, imRows, imCols, ConnectivityNdir);
        }
        else
            process_mesh1(ByteImagePtr, filters[0], imRows, imCols, ConnectivityNdir);

        /* Set up graph */
        find_connected_sets(RegionGraphBaseVertexPtr, ByteImagePtr, imRows, imCols, ConnectivityNdir);
        find_adjacency_and_extrema(RegionGraphBaseVertexPtr, MinimaList, MaximaList, &Min_min_area,
                                   &Max_min_area, filters[0], imRows, imCols, ConnectivityNdir);
    }
    else
    {
        /* Set up graph */
        find_connected_sets(RegionGraphBaseVertexPtr, ByteImagePtr, imRows, imCols, ConnectivityNdir);
        find_adjacency_and_extrema(RegionGraphBaseVertexPtr, MinimaList, MaximaList, &Min_min_area,
                                   &Max_min_area, 0, imRows, imCols, ConnectivityNdir);
    }

    /* Loop over input parameters */
    for (k=0; k<num_mesh; k++)
    {   /* Max_min_area and Min_max_area are used to skip scales (meshes) that contain no signal */
        /* Get input parameters */
        max_mesh = meshes[k];
        filter = filters[k];
        output = outputs[k];

        /* Initialise granule buffer to -1 (which signifies elements that are not yet used */
        if (GranuleListBasePtr!=0 && output)
        {
            GranuleListCurrentPtr = GranuleListBasePtr;
            for (k=1; k<=4; k++)
                GranuleListBasePtr[GRANSIZE*10*imRows*imCols-k] = -1;
               // GranuleListBasePtr[GRANSIZE*imRows*imCols-k] = -1;
			//for (k=1; k<(GRANSIZE*10*imRows*imCols); k++)
				//GranuleListBasePtr[k]=-1;
        }
        else
            GranuleListCurrentPtr = 0;

        /* Apply processing with selected filter */
        done = FALSE;
        while (!done)
        {
            /* Calculate maximum mesh for selected filter */
            if      (filter==OPENING) min_area = Max_min_area;
            else if (filter==CLOSING) min_area = Min_min_area;
            else                      min_area = (Min_min_area<Max_min_area)
                                              ? Min_min_area : Max_min_area;

            /* Test whether finished */
            if (min_area>0 && min_area<=max_mesh && min_area<imRows*imCols)
            {
                /* Apply filter at current mesh */
                if (filter==OPENCLOSE)
                {   // two passes at each scale
                    parse_extrema_list(RegionGraphBaseVertexPtr, MinimaList, MaximaList, &Min_min_area,
                        &Max_min_area, min_area, OPENING, offsets, &GranuleListCurrentPtr, GranuleCounter,ratio, minarea, maxarea, minamp);
                    parse_extrema_list(RegionGraphBaseVertexPtr, MinimaList, MaximaList, &Min_min_area,
                        &Max_min_area, min_area, CLOSING, offsets, &GranuleListCurrentPtr, GranuleCounter,ratio, minarea, maxarea, minamp);
                }
                else if (filter==CLOSEOPEN)
                {   // two passes at each scale
                    parse_extrema_list(RegionGraphBaseVertexPtr, MinimaList, MaximaList, &Min_min_area,
                        &Max_min_area, min_area, CLOSING, offsets, &GranuleListCurrentPtr, GranuleCounter,ratio, minarea, maxarea, minamp);
                    parse_extrema_list(RegionGraphBaseVertexPtr, MinimaList, MaximaList, &Min_min_area,
                        &Max_min_area, min_area, OPENING, offsets, &GranuleListCurrentPtr, GranuleCounter,ratio, minarea, maxarea, minamp);
                }
                else // single pass either of 'm', 'o', 'm' sieves
                    parse_extrema_list(RegionGraphBaseVertexPtr, MinimaList, MaximaList, &Min_min_area,
                        &Max_min_area, min_area, filter, offsets, &GranuleListCurrentPtr, GranuleCounter,ratio, minarea, maxarea, minamp);
            }
            else
                done = TRUE;
        }

        /* Generate low pass image for bandpass outputs */
        if (BandPassOutput1!=0)
        {
			Btmp = BandPassOutput1; 
			BandPassOutput1 = BandPassOutput2; 
			BandPassOutput2 = Btmp;
            make_lowpass_image_uint8(BandPassOutput1, RegionGraphBaseVertexPtr, imRows*imCols);
        }

        if (output>0)
            create_matlab_output((plhs+output-1), out_type, RegionGraphBaseVertexPtr, BandPassOutput1, BandPassOutput2,
                                 GranuleListBasePtr, GranuleListCurrentPtr, GranuleCounter, PicElementPtr, imRows, imCols, ConnectivityNdir);
    }
}


/****************************************************************************/

/* PARSE EXTREMA LIST FOR EXTREMA OF SPECIFIED AREA */

void parse_extrema_list(REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, 
						EXTREMAL_VERTEX *StartOfMinimaList, EXTREMAL_VERTEX *StartOfMaximaList,
                        int32 *Min_min_area, int32 *Max_min_area, int32 area,
                        int32 filter, int32 *offsets, int32 **GranuleListAlsoCurrentPtr, int32 *GranuleCounter,
						double ratio, int32 minarea, int32 maxarea, int32 minamp)
{
    int32 new_area, Min_root_PicElementPtr, Max_root_PicElementPtr;
    EXTREMAL_VERTEX *MinimaList_ptr, *MaximaList_ptr, *Minima_previous_extrema, *Maxima_previous_extrema;
    REGION_GRAPH_VERTEX *RegionGraphTemporaryVertexPtr;
    pointer Min_pos, Max_pos;

    Minima_previous_extrema = 0;
    Maxima_previous_extrema = 0;

    if (filter!=OPENING && *Min_min_area==area)
    {
        *Min_min_area = 0;
        Min_pos = 0;
    }
    else Min_pos = -1;

    if (filter!=CLOSING && *Max_min_area==area)
    {
        *Max_min_area = 0;
        Max_pos = 0;
    }
    else Max_pos = -1;

    /* Check whether first entry is valid */
    if (Min_pos>=0 && StartOfMinimaList->root_PicElementPtr < 0)
    {
        Minima_previous_extrema = StartOfMinimaList;
        Min_pos = StartOfMinimaList->NextExtremum;
    }
    if (Max_pos>=0 && StartOfMaximaList->root_PicElementPtr < 0)
    {
        Maxima_previous_extrema = StartOfMaximaList;
        Max_pos = StartOfMaximaList->NextExtremum;
    }

    /* Get root PicElementPtr position for first minimum and maximum in lists */
    if (Min_pos>=0)
        Min_root_PicElementPtr = (MinimaList_ptr=StartOfMinimaList+Min_pos)->root_PicElementPtr;
    if (Max_pos>=0)
        Max_root_PicElementPtr = (MaximaList_ptr=StartOfMaximaList+Max_pos)->root_PicElementPtr;

    /* Loop through maxima and minima lists */
    while (Min_pos>=0 || Max_pos>=0)
    {
        /* Process maxima and minima in order of root PicElementPtr position */
        if (Min_pos>=0 && (Max_pos<0 || Min_root_PicElementPtr<=Max_root_PicElementPtr))
        {
            /* Test area of extremum */
            RegionGraphTemporaryVertexPtr = RegionGraphBaseVertexPtr+Min_root_PicElementPtr;
            if ((new_area = -(RegionGraphTemporaryVertexPtr->root_PicElementPtr)) <= area)
            {
                if (new_area>0)
                    new_area = process_extremum(RegionGraphBaseVertexPtr, &RegionGraphTemporaryVertexPtr, 
					offsets, GranuleListAlsoCurrentPtr, GranuleCounter, ratio, minarea, maxarea, minamp);

                if (new_area<=0)
                {
                    /* Delete extremum from list */
                    if (Minima_previous_extrema)
                        Minima_previous_extrema->NextExtremum = MinimaList_ptr->NextExtremum;
                    else
                    {
                        MinimaList_ptr->root_PicElementPtr = -1;
                        Minima_previous_extrema = MinimaList_ptr;
                    }
                }
                else
                {
                    Minima_previous_extrema = MinimaList_ptr;
                    MinimaList_ptr->root_PicElementPtr = RegionGraphTemporaryVertexPtr-RegionGraphBaseVertexPtr;
                }
            }
            else
                Minima_previous_extrema = MinimaList_ptr;

            /* Update minimum area */
            if ((new_area<*Min_min_area || *Min_min_area==0) && new_area>0)
                *Min_min_area = new_area;

            /* Goto next extremum */
            if ((Min_pos = MinimaList_ptr->NextExtremum) >= 0)
                Min_root_PicElementPtr = (MinimaList_ptr=StartOfMinimaList+Min_pos)->root_PicElementPtr;
        }
        else   /* Process maxima */
        {
            /* Test area of extremum */
            RegionGraphTemporaryVertexPtr = RegionGraphBaseVertexPtr+Max_root_PicElementPtr;
            if ((new_area = -(RegionGraphTemporaryVertexPtr->root_PicElementPtr)) <= area)
            {
                if (new_area>0)
                    new_area = process_extremum(RegionGraphBaseVertexPtr, &RegionGraphTemporaryVertexPtr, 
					offsets, GranuleListAlsoCurrentPtr, GranuleCounter, ratio, minarea, maxarea, minamp);

                if (new_area<=0)
                {
                    /* Delete extremum from list */
                    if (Maxima_previous_extrema)
                        Maxima_previous_extrema->NextExtremum = MaximaList_ptr->NextExtremum;
                    else
                    {
                        MaximaList_ptr->root_PicElementPtr = -1;
                        Maxima_previous_extrema = MaximaList_ptr;
                    }
                }
                else
                {
                    Maxima_previous_extrema = MaximaList_ptr;
                    MaximaList_ptr->root_PicElementPtr = RegionGraphTemporaryVertexPtr-RegionGraphBaseVertexPtr;
                }
            }
            else
                Maxima_previous_extrema = MaximaList_ptr;

            /* Update minimum area */
            if ((new_area<*Max_min_area || *Max_min_area==0) && new_area>0)
                *Max_min_area = new_area;

            /* Goto next extremum */
            if ((Max_pos = MaximaList_ptr->NextExtremum) >= 0)
                Max_root_PicElementPtr = (MaximaList_ptr=StartOfMaximaList+Max_pos)->root_PicElementPtr;
        }
    }
}


/****************************************************************************/

/* PROCESS AN EXTREMUM */

int32 process_extremum(REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, 
	                   REGION_GRAPH_VERTEX **RegionGraphBaseReturnedVertexPtr, int32 *offsets,
                       int32 **GranuleListAlsoCurrentPtr, int32 *GranuleCounter, double ratio, int32 minarea, int32 maxarea, int32 minamp)
{
    #define LT_INIT -1
    #define GT_INIT 256

    int32 j, bit, num_eq, lt_num_eq, gt_num_eq, lt_val, gt_val, ext_val,
          adj_val, *offs, delta_area, delta_area2;
	double last_area;
    pointer pos, new_root_PicElementPtr, adjacencyRoot_ptr, lt_root[MAXEQ], gt_root[MAXEQ], current_area;
    uint8 flags, dirns;
    REGION_GRAPH_VERTEX *ListPicElementVertices_ptr, *adjacency_ptr,
		*adjacencyRoot_ptr_ptr, *LastPicElementPtr, *ptr, *RegionGraphTemporaryVertexPtr;


    /**********************************************************************/

    /* Find adjacent region with closest value */

    /* Prepare to find closest value adjacent region */
    RegionGraphTemporaryVertexPtr = *RegionGraphBaseReturnedVertexPtr;
    ext_val = RegionGraphTemporaryVertexPtr->value;
    pos = RegionGraphTemporaryVertexPtr-RegionGraphBaseVertexPtr;
    lt_val = LT_INIT;
    gt_val = GT_INIT;
    ListPicElementVertices_ptr = 0;
    LastPicElementPtr = 0;
    check_lists(1,0);

    /* Add current region to list so duplicates are deleted */
    check_lists(0,RegionGraphTemporaryVertexPtr-RegionGraphBaseVertexPtr);

    /* Loop over the distinct boundary PictureElement */
    while (pos>=0)
    {
        ListPicElementVertices_ptr = RegionGraphBaseVertexPtr+pos;
        dirns = flags = ListPicElementVertices_ptr->AdjacencyFlags;
        offs = offsets;
        bit = 1;

        /* Loop over possible adjacent directions */
        while (dirns)
        {
            if (dirns & 1)
            {
                /* Get value of adjacent region */
                adjacency_ptr = ListPicElementVertices_ptr+*offs;
                adjacencyRoot_ptr = adjacency_ptr->root_PicElementPtr;
                adjacencyRoot_ptr = (adjacencyRoot_ptr>=0) ? adjacencyRoot_ptr : adjacency_ptr-RegionGraphBaseVertexPtr;

                /* Check whether this is a duplicate */
                if (!check_lists(0, adjacencyRoot_ptr))
                    flags ^= bit;
                else
                {
                    /* Record closest values above and below extrema */
                    adj_val = (adjacencyRoot_ptr+RegionGraphBaseVertexPtr)->value;
                    if (adj_val<=ext_val)
                    {
                        if (adj_val>lt_val)
                        {
                            lt_val = adj_val;
                            lt_root[0] = adjacencyRoot_ptr;
                            lt_num_eq = 1;
                        }
                        else if (adj_val==lt_val && lt_num_eq<MAXEQ)
                            lt_root[lt_num_eq++] = adjacencyRoot_ptr;
                    }
                    else
                    {
                        if (adj_val<gt_val)
                        {
                            gt_val = adj_val;
                            gt_root[0] = adjacencyRoot_ptr;
                            gt_num_eq = 1;
                        }
                        else if (adj_val==gt_val && gt_num_eq<MAXEQ)
                            gt_root[gt_num_eq++] = adjacencyRoot_ptr;
                    }
                }
            }
            bit<<=1;
            dirns>>=1;
            offs++;
        }

        /* Update flags, deleting entry if necessary */
        if (!flags && LastPicElementPtr)
            LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = ListPicElementVertices_ptr->OffsetToNextBoundaryPicElementPtr;
        else
        {
            ListPicElementVertices_ptr->AdjacencyFlags = flags;
            LastPicElementPtr = ListPicElementVertices_ptr;
        }

        /* Goto next boundary PicElementPtr */
        pos = ListPicElementVertices_ptr->OffsetToNextBoundaryPicElementPtr;
    }

    /* Return if region is not an extremum */
    if (lt_val!=LT_INIT && gt_val!=GT_INIT && lt_val!=ext_val)
        return (0);


    /**********************************************************************/

    /* Merge PicElementPtr lists for extremum and adjacent regions */

    new_root_PicElementPtr = RegionGraphTemporaryVertexPtr-RegionGraphBaseVertexPtr;
    if (lt_val!=LT_INIT)
    {
        RegionGraphTemporaryVertexPtr->value = (uint8)lt_val;
        num_eq = lt_num_eq;
    }
    else
    {
        RegionGraphTemporaryVertexPtr->value = (uint8)gt_val;
        num_eq = gt_num_eq;
    }

    /* Store granule */
    if (*GranuleListAlsoCurrentPtr && ext_val!=RegionGraphTemporaryVertexPtr->value)
    {
		// look ahead at the regions we are about to merge into the extremum
		/* Loop over adjacent regions to merge to discover the change in area */
		delta_area=0;
		current_area= (double) -RegionGraphTemporaryVertexPtr->root_PicElementPtr;
		for (j=0; j<num_eq; j++)
		{
			/* Get information for closest adjacent region */
			adjacencyRoot_ptr = (lt_val!=LT_INIT) ? lt_root[j] : gt_root[j];
			adjacencyRoot_ptr_ptr = adjacencyRoot_ptr+RegionGraphBaseVertexPtr;

            /* Update area */
            delta_area += -adjacencyRoot_ptr_ptr->root_PicElementPtr;
		}
		//last_area=100*delta_area/(double)current_area;// to allow us to use int32 to pass value back to Matlab
		if (((int32)100.0*delta_area/(double)current_area>(100.0*ratio))&&
		//if ((delta_area>ratio) //&&
			(current_area>=minarea)&&			
			//(current_area<maxarea)//&&
			(abs(ext_val - RegionGraphTemporaryVertexPtr->value)>(minamp*256/100))
			)
		{
			*GranuleCounter+=1;

			//if ((ext_val - RegionGraphTemporaryVertexPtr->value)<0)
			//	mexPrintf(ext_val - RegionGraphTemporaryVertexPtr->value);

			*((*GranuleListAlsoCurrentPtr)++) = current_area; //-(RegionGraphTemporaryVertexPtr->root_PicElementPtr); // area - used to calculate offset to next granule
			*((*GranuleListAlsoCurrentPtr)++) = ext_val - RegionGraphTemporaryVertexPtr->value;
			*((*GranuleListAlsoCurrentPtr)++) = ext_val;
			*((*GranuleListAlsoCurrentPtr)++) = delta_area;
			*((*GranuleListAlsoCurrentPtr)++) = (int32) current_area; //last_area;
			*((*GranuleListAlsoCurrentPtr)++) = new_root_PicElementPtr;
			pos = RegionGraphTemporaryVertexPtr-RegionGraphBaseVertexPtr;
			while (pos>=0)
			{
				if (**GranuleListAlsoCurrentPtr==-1)
					mexErrMsgTxt("Out of space for list of granules");
				*((*GranuleListAlsoCurrentPtr)++) = pos;
				pos = (RegionGraphBaseVertexPtr+pos)->next_PicElementPtr;
			}
		}
    }
	//RegionGraphTemporaryVertexPtr->last_area=(double)(0.5*RegionGraphTemporaryVertexPtr->last_area) +
	//	(1-0.5)*(double)RegionGraphTemporaryVertexPtr->root_PicElementPtr;
	RegionGraphTemporaryVertexPtr->last_area=(double)-RegionGraphTemporaryVertexPtr->root_PicElementPtr;
    /* Loop over adjacent regions to merge */
	delta_area2=0;
    for (j=0; j<num_eq; j++)
    {
        /* Get information for closest adjacent region */
        adjacencyRoot_ptr = (lt_val!=LT_INIT) ? lt_root[j] : gt_root[j];
        adjacencyRoot_ptr_ptr = adjacencyRoot_ptr+RegionGraphBaseVertexPtr;

        //if (adjacencyRoot_ptr_ptr->root_PicElementPtr >= RegionGraphTemporaryVertexPtr->root_PicElementPtr) // add smaller region to larger
		if (true) // add adjacent to extremum
		{  
            /* Update area */
			//delta_area2+=-adjacencyRoot_ptr_ptr->root_PicElementPtr;
			RegionGraphTemporaryVertexPtr->last_area          +=-adjacencyRoot_ptr_ptr->root_PicElementPtr;
            RegionGraphTemporaryVertexPtr->root_PicElementPtr += adjacencyRoot_ptr_ptr->root_PicElementPtr;

            /* Modify root PictureElement of adjacent region */
            (ptr=adjacencyRoot_ptr_ptr)->root_PicElementPtr = new_root_PicElementPtr;
            while (ptr->next_PicElementPtr>=0)
            {
                ptr = RegionGraphBaseVertexPtr+ptr->next_PicElementPtr;
                ptr->root_PicElementPtr = new_root_PicElementPtr;
            }

            /* Insert adjacent region after root PicElementPtr of extrema */
            ptr->next_PicElementPtr = RegionGraphTemporaryVertexPtr->next_PicElementPtr;
            RegionGraphTemporaryVertexPtr->next_PicElementPtr = adjacencyRoot_ptr_ptr-RegionGraphBaseVertexPtr;

            /* Add adjacent boundary PictureElement after extremum boundary PictureElement */
            LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = adjacencyRoot_ptr;
            if (j!=num_eq-1)
            {
                LastPicElementPtr = RegionGraphBaseVertexPtr+LastPicElementPtr->OffsetToNextBoundaryPicElementPtr;
                while (LastPicElementPtr->OffsetToNextBoundaryPicElementPtr>=0)
                    LastPicElementPtr = RegionGraphBaseVertexPtr+LastPicElementPtr->OffsetToNextBoundaryPicElementPtr;
            }
        }
        else
        {
            /* Update area */
            adjacencyRoot_ptr_ptr->root_PicElementPtr += RegionGraphTemporaryVertexPtr->root_PicElementPtr;

            /* Modify root PictureElement of extremum */
            (ptr=RegionGraphTemporaryVertexPtr)->root_PicElementPtr = adjacencyRoot_ptr;
            while (ptr->next_PicElementPtr>=0)
            {
                ptr = RegionGraphBaseVertexPtr+ptr->next_PicElementPtr;
                ptr->root_PicElementPtr = adjacencyRoot_ptr;
            }

            /* Insert extremum after root PicElementPtr of adjacent region */
            ptr->next_PicElementPtr = adjacencyRoot_ptr_ptr->next_PicElementPtr;
            adjacencyRoot_ptr_ptr->next_PicElementPtr = new_root_PicElementPtr;

            /* Add extremum boundary PictureElement */
            if (j!=num_eq-1)
            {
                /* Add extremum boundary PictureElement after adjacent boundary PictureElement*/
                ptr = adjacencyRoot_ptr_ptr;
                while (ptr->OffsetToNextBoundaryPicElementPtr>=0)
                    ptr = RegionGraphBaseVertexPtr+ptr->OffsetToNextBoundaryPicElementPtr;
                ptr->OffsetToNextBoundaryPicElementPtr = new_root_PicElementPtr;
            }
            else
            {
                /* Insert extremum boundary PictureElement after adjacent root PicElementPtr */
                LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = adjacencyRoot_ptr_ptr->OffsetToNextBoundaryPicElementPtr;
                adjacencyRoot_ptr_ptr->OffsetToNextBoundaryPicElementPtr = new_root_PicElementPtr;
            }

            /* Make adjacent region the extremum */
            RegionGraphTemporaryVertexPtr = adjacencyRoot_ptr_ptr;
            new_root_PicElementPtr = adjacencyRoot_ptr;
        }
    }
	//RegionGraphTemporaryVertexPtr->last_area=delta_area2;

    *RegionGraphBaseReturnedVertexPtr = RegionGraphTemporaryVertexPtr;
    return (-(RegionGraphTemporaryVertexPtr->root_PicElementPtr));
}


/****************************************************************************/

/* FIND CONNECTED-SETS */
/* Flood fill regions of neighbours with the same image value
   Return, REGION_GRAPH_VERTEX structure with some (not all) fields instantiated */

void find_connected_sets(REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, uint8 *StartOfImage, int32 imRows, int32 imCols,
                         int32 ConnectivityNdir)
{

    int32 m, n, m0, n0, val, dir1, dir2, dir3,dir4;
    uint8 *TempImagePtr, *Temp2ImagePtr, last_dirn, other_regions;
    pointer pos, next_PicElementPtr;
    REGION_GRAPH_VERTEX *RegionGraphTemporaryVertexPtr, *ListPicElementVertices_ptr, *LastPicElementPtr;

    if (ConnectivityNdir==8)
    {
        dir1 = (1+16+32);
        dir2 = (2+64+128);
        dir3 = (4+16+64);
        dir4 = (8+32+128);
    }
    else
    {
        dir1 = (1+4+8);
        dir2 = (2+4+8);
        dir3 = (1+2+4);
        dir4 = (1+2+8);
    }

    RegionGraphTemporaryVertexPtr = RegionGraphBaseVertexPtr;
    TempImagePtr = Temp2ImagePtr = StartOfImage;
    pos = 0;

	/* The following is basically a flood fill in which the RegionGraphBaseVertexPtr data structure is built */

    for (n=0; n<imCols; n++)
    {
        for (m=0; m<imRows; m++, pos++, TempImagePtr++, RegionGraphTemporaryVertexPtr++)
        {
            /* Test whether already used */
            if (!RegionGraphTemporaryVertexPtr->OffsetToNextBoundaryPicElementPtr)
            {
                /* Add root PicElementPtr */
                RegionGraphTemporaryVertexPtr->root_PicElementPtr = pos; // the first picture element in the current region
                RegionGraphTemporaryVertexPtr->OffsetToNextBoundaryPicElementPtr = 1; // might be updated
                other_regions  = 0; // byte used to flag directions of neighbouring regions (with different picture element values)
                LastPicElementPtr   = RegionGraphTemporaryVertexPtr;
                val = RegionGraphTemporaryVertexPtr->value = *TempImagePtr; // image value of current region

                /* Try 4 valid directions to move from root PicElementPtr */
                if (m!=imRows-1) // stop it looking beyond the bottom of a column
                {
                    if (val==*(TempImagePtr+1))
                    {   // add this picture element to the region 
                        LastPicElementPtr->next_PicElementPtr = pos+1;
                        LastPicElementPtr = RegionGraphTemporaryVertexPtr+1; // move on to the next pixel and point back to previous element of the same region
                        LastPicElementPtr->AdjacencyFlags = 2;  // down
                        LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = ((m+1)<<12) | (n+1);
                    }
                    else other_regions |= 2;
                }
                if (n!=imCols-1) // stop it looking beyound the rightmost row
                {
                    if (val==*(TempImagePtr+imRows))
                    {
                        LastPicElementPtr->next_PicElementPtr = pos+imRows;
                        LastPicElementPtr = RegionGraphTemporaryVertexPtr+imRows;
                        LastPicElementPtr->AdjacencyFlags = 8; // right
                        LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = ((m)<<12) | (n+1+1);
                    }
                    else other_regions |= 8;
                }
                if (ConnectivityNdir==8)
                {
                    if (m!=0 && n!=imCols-1)
                    {
                        if (val==*(TempImagePtr-1+imRows))
                        {
                            LastPicElementPtr->next_PicElementPtr = pos-1+imRows;
                            LastPicElementPtr = RegionGraphTemporaryVertexPtr-1+imRows;
                            LastPicElementPtr->AdjacencyFlags = 32;
                            LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = ((m-1)<<12) | (n+1+1);
                        }
                        else other_regions |= 32; // up-right?
                    }
                    if (m!=imRows-1 && n!=imCols-1)
                    {
                        if (val==*(TempImagePtr+1+imRows))
                        {
                            LastPicElementPtr->next_PicElementPtr = pos+1+imRows;
                            LastPicElementPtr = RegionGraphTemporaryVertexPtr+1+imRows;
                            LastPicElementPtr->AdjacencyFlags = 128;
                            LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = ((m+1)<<12) | (n+1+1);
                        }
                        else other_regions |= 128; /// down-right?
                    }
                }

                /* Check other directions for other regions */
                if (m!=0 && val!=*(TempImagePtr-1))  other_regions |= 1;
                if (n!=0 && val!=*(TempImagePtr-imRows))  other_regions |= 4;
                if (ConnectivityNdir==8)
                {
                    if (m!=0   && n!=0 && val!=*(TempImagePtr-1-imRows)) other_regions |= 16;
                    if (m!=imRows-1 && n!=0 && val!=*(TempImagePtr+1-imRows)) other_regions |= 64;
                }

                /* Add end of LastPicElementPtr list marker */
                LastPicElementPtr->next_PicElementPtr = -1;

                /* Store directions to other regions */
                RegionGraphTemporaryVertexPtr->AdjacencyFlags = other_regions;

                /* Loop over all PictureElement added to region */
                next_PicElementPtr = RegionGraphTemporaryVertexPtr->next_PicElementPtr;
                while (next_PicElementPtr >= 0)
                {
                    ListPicElementVertices_ptr = RegionGraphBaseVertexPtr + next_PicElementPtr;
                    Temp2ImagePtr = StartOfImage + next_PicElementPtr;
                    val = *Temp2ImagePtr;

                    /* Set root PicElementPtr */
                    ListPicElementVertices_ptr->root_PicElementPtr = pos;

                    /* Get previous direction */
                    last_dirn = ListPicElementVertices_ptr->AdjacencyFlags;
                    m0 = ListPicElementVertices_ptr->OffsetToNextBoundaryPicElementPtr >> 12;
                    n0 = (ListPicElementVertices_ptr->OffsetToNextBoundaryPicElementPtr & 4095) - 1;
                    other_regions = 0;

                    /* Consider all possible directions */
                    if ((last_dirn & dir1) && m0!=0)
                    {
                        if (val==*(Temp2ImagePtr-1))
                        {
                            if (!(ListPicElementVertices_ptr-1)->OffsetToNextBoundaryPicElementPtr)
                            {
                                LastPicElementPtr->next_PicElementPtr = next_PicElementPtr-1;
                                LastPicElementPtr = ListPicElementVertices_ptr-1;
                                LastPicElementPtr->AdjacencyFlags = 1;
                                LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = ((m0-1)<<12)|(n0+1);
                            }
                        }
                        else other_regions |= 1;
                    }
                    if ((last_dirn & dir2) && m0!=imRows-1)
                    {
                        if (val==*(Temp2ImagePtr+1))
                        {
                            if (!(ListPicElementVertices_ptr+1)->OffsetToNextBoundaryPicElementPtr)
                            {
                                LastPicElementPtr->next_PicElementPtr = next_PicElementPtr+1;
                                LastPicElementPtr = ListPicElementVertices_ptr+1;
                                LastPicElementPtr->AdjacencyFlags = 2;
                                LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = ((m0+1)<<12)|(n0+1);
                            }
                        }
                        else other_regions |= 2;
                    }
                    if ((last_dirn & dir3) && n0!=0)
                    {
                        if (val==*(Temp2ImagePtr-imRows))
                        {
                            if (!(ListPicElementVertices_ptr-imRows)->OffsetToNextBoundaryPicElementPtr)
                            {
                                LastPicElementPtr->next_PicElementPtr = next_PicElementPtr-imRows;
                                LastPicElementPtr = ListPicElementVertices_ptr-imRows;
                                LastPicElementPtr->AdjacencyFlags = 4;
                                LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = ((m0)<<12)|(n0-1+1);
                            }
                        }
                        else other_regions |= 4;
                    }
                    if ((last_dirn & dir4) && n0!=imCols-1)
                    {
                        if (val==*(Temp2ImagePtr+imRows))
                        {
                            if (!(ListPicElementVertices_ptr+imRows)->OffsetToNextBoundaryPicElementPtr)
                            {
                                LastPicElementPtr->next_PicElementPtr = next_PicElementPtr+imRows;
                                LastPicElementPtr = ListPicElementVertices_ptr+imRows;
                                LastPicElementPtr->AdjacencyFlags = 8;
                                LastPicElementPtr->OffsetToNextBoundaryPicElementPtr = ((m0)<<12)|(n0+1+1);
                            }
                        }
                        else other_regions |= 8;
                    }
                    if (ConnectivityNdir==8)
                    {
                        if ((last_dirn & (1+4+16+32+64)) && m0!=0 && n0!=0)
                        {
                            if (val==*(Temp2ImagePtr-1-imRows))
                            {
                                if (!(ListPicElementVertices_ptr-1-imRows)->OffsetToNextBoundaryPicElementPtr)
                                {
                                    LastPicElementPtr->next_PicElementPtr = next_PicElementPtr-1-imRows;
                                    LastPicElementPtr = ListPicElementVertices_ptr-1-imRows;
                                    LastPicElementPtr->AdjacencyFlags = 16;
                                    LastPicElementPtr->OffsetToNextBoundaryPicElementPtr =((m0-1)<<12)|(n0);
                                }
                            }
                            else other_regions |= 16;
                        }
                        if ((last_dirn & (1+8+16+32+128)) && m0!=0 && n0!=imCols-1)
                        {
                            if (val==*(Temp2ImagePtr-1+imRows))
                            {
                                if (!(ListPicElementVertices_ptr-1+imRows)->OffsetToNextBoundaryPicElementPtr)
                                {
                                    LastPicElementPtr->next_PicElementPtr = next_PicElementPtr-1+imRows;
                                    LastPicElementPtr = ListPicElementVertices_ptr-1+imRows;
                                    LastPicElementPtr->AdjacencyFlags = 32;
                                    LastPicElementPtr->OffsetToNextBoundaryPicElementPtr =((m0-1)<<12)|(n0+2);
                                }
                            }
                            else other_regions |= 32;
                        }
                        if ((last_dirn & (2+4+16+64+128)) && m0!=imRows-1 && n0!=0)
                        {
                            if (val==*(Temp2ImagePtr+1-imRows))
                            {
                                if (!(ListPicElementVertices_ptr+1-imRows)->OffsetToNextBoundaryPicElementPtr)
                                {
                                    LastPicElementPtr->next_PicElementPtr = next_PicElementPtr+1-imRows;
                                    LastPicElementPtr = ListPicElementVertices_ptr+1-imRows;
                                    LastPicElementPtr->AdjacencyFlags = 64;
                                    LastPicElementPtr->OffsetToNextBoundaryPicElementPtr =((m0+1)<<12)|(n0);
                                }
                            }
                            else other_regions |= 64;
                        }
                        if ((last_dirn & (2+8+32+64+128)) && m0!=imRows-1 &&n0!=imCols-1)
                        {
                            if (val==*(Temp2ImagePtr+1+imRows))
                            {
                                if (!(ListPicElementVertices_ptr+1+imRows)->OffsetToNextBoundaryPicElementPtr)
                                {
                                    LastPicElementPtr->next_PicElementPtr = next_PicElementPtr+1+imRows;
                                    LastPicElementPtr = ListPicElementVertices_ptr+1+imRows;
                                    LastPicElementPtr->AdjacencyFlags = 128;
                                    LastPicElementPtr->OffsetToNextBoundaryPicElementPtr =((m0+1)<<12)|(n0+2);
                                }
                            }
                            else other_regions |= 128;
                        }
                    }

                    /* Add end of list marker */
                    LastPicElementPtr->next_PicElementPtr = -1;

                    /* Store directions of other regions */
                    ListPicElementVertices_ptr->AdjacencyFlags = other_regions;

                    /* Goto next PicElementPtr */
                    next_PicElementPtr = ListPicElementVertices_ptr->next_PicElementPtr;

                 }
            }
        }
    }
}


/***************************************************************************/

/* FIND DISTINCT BOUNDARY PICTURE_ELEMENTS FOR EACH CONNECTED-SET */
/* Return:
   MinimaList, MaximaList
   Adjacency field of REGION_GRAPH_VERTEX i.e. */
void find_adjacency_and_extrema(REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, EXTREMAL_VERTEX *StartOfMinimaList,
        EXTREMAL_VERTEX *StartOfMaximaList, int32 *Min_min_area, int32 *Max_min_area,
        int32 filter, int32 imRows, int32 imCols, int32 ConnectivityNdir)
{
    int32 area, num_min, num_max, val, sign;
    uint8 dirns, flags;
    REGION_GRAPH_VERTEX *RegionGraphTemporaryVertexPtr, *ListPicElementVertices_ptr, *adjacency_ptr, *prev_adjacency_ptr;
    EXTREMAL_VERTEX *MinimaList_ptr, *MaximaList_ptr;
    pointer adjacencyRoot_ptr, pos;

    RegionGraphTemporaryVertexPtr = RegionGraphBaseVertexPtr;
    MinimaList_ptr = StartOfMinimaList;
    MaximaList_ptr = StartOfMaximaList;
    num_min = 0;
    num_max = 0;
    *Min_min_area = 0;
    *Max_min_area = 0;

    for (pos=0; pos<imRows*imCols; pos++, RegionGraphTemporaryVertexPtr++)
    {
        /* Test if current node is root node */
        if (RegionGraphTemporaryVertexPtr->root_PicElementPtr == pos)
        {

            /* Trap special case of single PicElementPtr region */
            if (RegionGraphTemporaryVertexPtr->next_PicElementPtr<0 && filter==MEDIAN)
            {
                RegionGraphTemporaryVertexPtr->root_PicElementPtr = -1;
                RegionGraphTemporaryVertexPtr->OffsetToNextBoundaryPicElementPtr = -1;
            }
            else
            {
                area = 0;
                sign = 0;
                check_lists(1, 0);

                /* Loop over all PictureElement in connected-set */
                ListPicElementVertices_ptr = RegionGraphTemporaryVertexPtr;
                prev_adjacency_ptr = 0;
                val = RegionGraphTemporaryVertexPtr->value;
                do
                {
                    area++;

                    /* Find new regions adjacent to current PicElementPtr */
                    dirns = ListPicElementVertices_ptr->AdjacencyFlags;
                    flags = 0;

                    if (dirns & 1)
                    {
                        adjacency_ptr  = ListPicElementVertices_ptr-1;
                        adjacencyRoot_ptr = (adjacency_ptr->root_PicElementPtr>=0) ?
                                    adjacency_ptr->root_PicElementPtr:adjacency_ptr-RegionGraphBaseVertexPtr;
                        if (check_lists(0, adjacencyRoot_ptr))
                        {
                            flags |= 1;
                            sign |= ((int32)(RegionGraphBaseVertexPtr+adjacencyRoot_ptr)->value>val)+1;
                        }
                    }

                    if (dirns & 2)
                    {
                        adjacency_ptr  = ListPicElementVertices_ptr+1;
                        adjacencyRoot_ptr = (adjacency_ptr->root_PicElementPtr>=0) ?
                                    adjacency_ptr->root_PicElementPtr:adjacency_ptr-RegionGraphBaseVertexPtr;
                        if (check_lists(0, adjacencyRoot_ptr))
                        {
                            flags |= 2;
                            sign |= ((int32)(RegionGraphBaseVertexPtr+adjacencyRoot_ptr)->value>val)+1;
                        }
                    }

                    if (dirns & 4)
                    {
                        adjacency_ptr  = ListPicElementVertices_ptr-imRows;
                        adjacencyRoot_ptr = (adjacency_ptr->root_PicElementPtr>=0) ?
                                    adjacency_ptr->root_PicElementPtr:adjacency_ptr-RegionGraphBaseVertexPtr;
                        if (check_lists(0, adjacencyRoot_ptr))
                        {
                            flags |= 4;
                            sign |= ((int32)(RegionGraphBaseVertexPtr+adjacencyRoot_ptr)->value>val)+1;
                        }
                    }

                    if (dirns & 8)
                    {
                        adjacency_ptr  = ListPicElementVertices_ptr+imRows;
                        adjacencyRoot_ptr = (adjacency_ptr->root_PicElementPtr>=0) ?
                                    adjacency_ptr->root_PicElementPtr:adjacency_ptr-RegionGraphBaseVertexPtr;
                        if (check_lists(0, adjacencyRoot_ptr))
                        {
                            flags |= 8;
                            sign |= ((int32)(RegionGraphBaseVertexPtr+adjacencyRoot_ptr)->value>val)+1;
                        }
                    }

                    if (ConnectivityNdir==8)
                    {
                        if (dirns & 16)
                        {
                            adjacency_ptr  = ListPicElementVertices_ptr-1-imRows;
                            adjacencyRoot_ptr = (adjacency_ptr->root_PicElementPtr>=0) ?
                                        adjacency_ptr->root_PicElementPtr:adjacency_ptr-RegionGraphBaseVertexPtr;
                            if (check_lists(0, adjacencyRoot_ptr))
                            {
                                flags |= 16;
                                sign |=((int32)(RegionGraphBaseVertexPtr+adjacencyRoot_ptr)->value>val)+1;
                            }
                        }

                        if (dirns & 32)
                        {
                            adjacency_ptr  = ListPicElementVertices_ptr-1+imRows;
                            adjacencyRoot_ptr = (adjacency_ptr->root_PicElementPtr>=0) ?
                                        adjacency_ptr->root_PicElementPtr:adjacency_ptr-RegionGraphBaseVertexPtr;
                            if (check_lists(0, adjacencyRoot_ptr))
                            {
                                flags |= 32;
                                sign |=((int32)(RegionGraphBaseVertexPtr+adjacencyRoot_ptr)->value>val)+1;
                            }
                        }

                        if (dirns & 64)
                        {
                            adjacency_ptr  = ListPicElementVertices_ptr+1-imRows;
                            adjacencyRoot_ptr = (adjacency_ptr->root_PicElementPtr>=0) ?
                                        adjacency_ptr->root_PicElementPtr:adjacency_ptr-RegionGraphBaseVertexPtr;
                            if (check_lists(0, adjacencyRoot_ptr))
                            {
                                flags |= 64;
                                sign |=((int32)(RegionGraphBaseVertexPtr+adjacencyRoot_ptr)->value>val)+1;
                            }
                        }

                        if (dirns & 128)
                        {
                            adjacency_ptr  = ListPicElementVertices_ptr+1+imRows;
                            adjacencyRoot_ptr = (adjacency_ptr->root_PicElementPtr>=0) ?
                                        adjacency_ptr->root_PicElementPtr:adjacency_ptr-RegionGraphBaseVertexPtr;
                            if (check_lists(0, adjacencyRoot_ptr))
                            {
                                flags |= 128;
                                sign |=((int32)(RegionGraphBaseVertexPtr+adjacencyRoot_ptr)->value>val)+1;
                            }
                        }
                    }

                    /* Set node data */
                    if (flags && prev_adjacency_ptr)
                        prev_adjacency_ptr->OffsetToNextBoundaryPicElementPtr = ListPicElementVertices_ptr-RegionGraphBaseVertexPtr;
                    if (flags || prev_adjacency_ptr==0)
                        prev_adjacency_ptr = ListPicElementVertices_ptr;
                    ListPicElementVertices_ptr->AdjacencyFlags = flags;

                    /* Goto next PicElementPtr in connected-set */
                    ListPicElementVertices_ptr = RegionGraphBaseVertexPtr+ListPicElementVertices_ptr->next_PicElementPtr;
                }
                while (ListPicElementVertices_ptr>=RegionGraphBaseVertexPtr);

                /* Store (-)area in pointer field of root PicElementPtr */
                RegionGraphTemporaryVertexPtr->root_PicElementPtr = (pointer)(-area);
                prev_adjacency_ptr->OffsetToNextBoundaryPicElementPtr = -1;

                /* Determine whether extremum */
                if (sign==2)
                {
                    if (MinimaList_ptr->NextExtremum == -1)
                        mexErrMsgTxt("Out of space for list of minima");
                    MinimaList_ptr->root_PicElementPtr = (RegionGraphTemporaryVertexPtr-RegionGraphBaseVertexPtr);
                    (MinimaList_ptr++)->NextExtremum = ++num_min;
                    if (area<*Min_min_area || *Min_min_area==0)
                        *Min_min_area = area;
                }
                else if (sign==1)
                {
                    if (MaximaList_ptr->NextExtremum == -1)
                        mexErrMsgTxt("Out of space for list of maxima");
                    MaximaList_ptr->root_PicElementPtr = (RegionGraphTemporaryVertexPtr-RegionGraphBaseVertexPtr);
                    (MaximaList_ptr++)->NextExtremum = ++num_max;
                    if (area<*Max_min_area || *Max_min_area==0)
                        *Max_min_area = area;
                }
            }
        }
    }

    /* Add end of list marker */
    if (num_min) (MinimaList_ptr-1)->NextExtremum = -1;
    if (num_max) (MaximaList_ptr-1)->NextExtremum = -1;
}


/****************************************************************************/

/* STORE ADJACENT REGIONS IN HASH TABLE */

int32 check_lists(bool reset, pointer root)
{
    static pointer lists[16][HASHLEN];
    static int32 lists_len[16];
    pointer *list;
    int32 k, hash, *len;

    if (!reset)
    {
        hash = ((root & 768)>>6) | (root & 3);
        list = lists[hash];
        for (k=0; k<lists_len[hash]; k++)
            if (*(list++) == root) return FALSE;
        *list = root;
        if ((++lists_len[hash]) == HASHLEN)
            mexErrMsgTxt("Out of space for list of adjacent regions");

        return TRUE;
    }
    else
    {
        len = lists_len;
        *(len++) = 0; *(len++) = 0; *(len++) = 0; *(len++) = 0;
        *(len++) = 0; *(len++) = 0; *(len++) = 0; *(len++) = 0;
        *(len++) = 0; *(len++) = 0; *(len++) = 0; *(len++) = 0;
        *(len++) = 0; *(len++) = 0; *(len++) = 0; *(len++) = 0;
        return FALSE;
    }
}


/****************************************************************************/

/* PROCESS RAW IMAGE AT MESH 1 */

/* Process mesh 1 */
void process_mesh1(uint8 *ByteImagePtr, int32 filter, int32 imRows, int32 imCols, int32 ConnectivityNdir)
{
    int32 m, n;
    uint8 val;

    for (n=0; n<imCols; n++)
    {
        for (m=0; m<imRows; m++, ByteImagePtr++)
        {
            val = *ByteImagePtr;

            /* Test whether edge PicElementPtr */
            if (m==0 || m==imRows-1 || n==0 || n==imCols-1)
            {
                /* Test if maximum */
                if (filter!=CLOSING &&
                    (m==0 || *(ByteImagePtr-1)<val) && (m==imRows-1 || *(ByteImagePtr+1)<val) &&
                    (n==0 || *(ByteImagePtr-imRows)<val) && (n==imCols-1 || *(ByteImagePtr+imRows)<val) &&
                    (ConnectivityNdir==4 || (
                    (m==0   || n==0   || *(ByteImagePtr-1-imRows)<val) &&
                    (m==0   || n==imCols-1 || *(ByteImagePtr-1+imRows)<val) &&
                    (m==imRows-1 || n==0   || *(ByteImagePtr+1-imRows)<val) &&
                    (m==imRows-1 || n==imCols-1 || *(ByteImagePtr+1+imRows)<val))))
                {
                    /* Find maximum adjacent PicElementPtr */
                    val = (m!=0) ? *(ByteImagePtr-1) : 0;
                    if (m!=imRows-1 && *(ByteImagePtr+1)>val)   val=*(ByteImagePtr+1);
                    if (n!=0   && *(ByteImagePtr-imRows)>val)   val=*(ByteImagePtr-imRows);
                    if (n!=imCols-1 && *(ByteImagePtr+imRows)>val)   val=*(ByteImagePtr+imRows);
                    if (ConnectivityNdir==8)
                    {
                        if (m!=0   && n!=0   && *(ByteImagePtr-1-imRows)>val) val=*(ByteImagePtr-1-imRows);
                        if (m!=0   && n!=imCols-1 && *(ByteImagePtr-1+imRows)>val) val=*(ByteImagePtr-1+imRows);
                        if (m!=imRows-1 && n!=0   && *(ByteImagePtr+1-imRows)>val) val=*(ByteImagePtr+1-imRows);
                        if (m!=imRows-1 && n!=imCols-1 && *(ByteImagePtr+1+imRows)>val) val=*(ByteImagePtr+1+imRows);
                    }

                    /* Substitute new value */
                    *ByteImagePtr = val;
                }

                /* Test if minimum */
                else if (filter!=OPENING &&
                    (m==0 || *(ByteImagePtr-1)>val) && (m==imRows-1 || *(ByteImagePtr+1)>val) &&
                    (n==0 || *(ByteImagePtr-imRows)>val) && (n==imCols-1 || *(ByteImagePtr+imRows)>val) &&
                    (ConnectivityNdir==4 || (
                    (m==0   || n==0   || *(ByteImagePtr-1-imRows)>val) &&
                    (m==0   || n==imCols-1 || *(ByteImagePtr-1+imRows)>val) &&
                    (m==imRows-1 || n==0   || *(ByteImagePtr+1-imRows)>val) &&
                    (m==imRows-1 || n==imCols-1 || *(ByteImagePtr+1+imRows)>val))))
                {
                    /* Find mimimum adjacent PicElementPtr */
                    val = (m!=0) ? *(ByteImagePtr-1) : 255;
                    if (m!=imRows-1 && *(ByteImagePtr+1)<val)   val=*(ByteImagePtr+1);
                    if (n!=0   && *(ByteImagePtr-imRows)<val)   val=*(ByteImagePtr-imRows);
                    if (n!=imCols-1 && *(ByteImagePtr+imRows)<val)   val=*(ByteImagePtr+imRows);
                    if (ConnectivityNdir==8)
                    {
                        if (m!=0   && n!=0   && *(ByteImagePtr-1-imRows)<val) val=*(ByteImagePtr-1-imRows);
                        if (m!=0   && n!=imCols-1 && *(ByteImagePtr-1+imRows)<val) val=*(ByteImagePtr-1+imRows);
                        if (m!=imRows-1 && n!=0   && *(ByteImagePtr+1-imRows)<val) val=*(ByteImagePtr+1-imRows);
                        if (m!=imRows-1 && n!=imCols-1 && *(ByteImagePtr+1+imRows)<val) val=*(ByteImagePtr+1+imRows);
                    }

                    /* Substitute new value */
                    *ByteImagePtr = val;
                }
            }
            else
            {
                /* Test if maximum */
                if (filter!=CLOSING &&
                    *(ByteImagePtr-1)<val && *(ByteImagePtr+1)<val && *(ByteImagePtr-imRows)<val &&
                    *(ByteImagePtr+imRows)<val && (ConnectivityNdir==4 || (
                    *(ByteImagePtr-1-imRows)<val && *(ByteImagePtr-1+imRows)<val &&
                    *(ByteImagePtr+1-imRows)<val && *(ByteImagePtr+1+imRows)<val)))
                {
                    /* Find maximum adjacent PicElementPtr */
                    val = *(ByteImagePtr-1);
                    if (*(ByteImagePtr+1)>val)   val=*(ByteImagePtr+1);
                    if (*(ByteImagePtr-imRows)>val)   val=*(ByteImagePtr-imRows);
                    if (*(ByteImagePtr+imRows)>val)   val=*(ByteImagePtr+imRows);
                    if (ConnectivityNdir==8)
                    {
                        if (*(ByteImagePtr-1-imRows)>val) val=*(ByteImagePtr-1-imRows);
                        if (*(ByteImagePtr-1+imRows)>val) val=*(ByteImagePtr-1+imRows);
                        if (*(ByteImagePtr+1-imRows)>val) val=*(ByteImagePtr+1-imRows);
                        if (*(ByteImagePtr+1+imRows)>val) val=*(ByteImagePtr+1+imRows);
                    }

                    /* Substitute new value */
                    *ByteImagePtr = val;
                }

                /* Test if minimum */
                else if (filter!=OPENING &&
                    *(ByteImagePtr-1)>val && *(ByteImagePtr+1)>val && *(ByteImagePtr-imRows)>val &&
                    *(ByteImagePtr+imRows)>val && (ConnectivityNdir==4 || (
                    *(ByteImagePtr-1-imRows)>val && *(ByteImagePtr-1+imRows)>val &&
                    *(ByteImagePtr+1-imRows)>val && *(ByteImagePtr+1+imRows)>val)))
                {
                    /* Find minimum adjacent PicElementPtr */
                    val = *(ByteImagePtr-1);
                    if (*(ByteImagePtr+1)<val)   val=*(ByteImagePtr+1);
                    if (*(ByteImagePtr-imRows)<val)   val=*(ByteImagePtr-imRows);
                    if (*(ByteImagePtr+imRows)<val)   val=*(ByteImagePtr+imRows);
                    if (ConnectivityNdir==8)
                    {
                        if (*(ByteImagePtr-1-imRows)<val) val=*(ByteImagePtr-1-imRows);
                        if (*(ByteImagePtr-1+imRows)<val) val=*(ByteImagePtr-1+imRows);
                        if (*(ByteImagePtr+1-imRows)<val) val=*(ByteImagePtr+1-imRows);
                        if (*(ByteImagePtr+1+imRows)<val) val=*(ByteImagePtr+1+imRows);
                    }

                    /* Substitute new value */
                    *ByteImagePtr = val;
                }
            }
        }
    }
}


/****************************************************************************/

/* COPY INPUT IMAGE TO BYTE ARRAY */

void copy_image_double(uint8 *Y, double *X, int32 PictureElement)
{
    int32 k, val;

    for (k=0; k<PictureElement; k++)
    {
        val = (int32)*(X++);
        if (val & (~255))
            mexErrMsgTxt("Data must be in range 0 to 255");
        *(Y++) = (uint8)val;
    }
}


void copy_image_uint8(uint8 *Y, uint8 *X, int32 PictureElement)
{
    int32 k;

    for (k=0; k<PictureElement; k++)
        *(Y++) = *(X++);
}


/****************************************************************************/

/* GENERATE LOW PASS IMAGE */

void make_lowpass_image_int16(int16 *Y, REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, int32 PictureElement)
{
    int32 k;
    REGION_GRAPH_VERTEX *RegionGraphTemporaryVertexPtr;

    RegionGraphTemporaryVertexPtr = RegionGraphBaseVertexPtr;

    /* Loop over all PictureElement */
    for (k=0; k<PictureElement; k++, RegionGraphTemporaryVertexPtr++, Y++)
        *Y = (RegionGraphTemporaryVertexPtr->root_PicElementPtr < 0) ?
            (int16)RegionGraphTemporaryVertexPtr->value : (int16)(RegionGraphBaseVertexPtr+RegionGraphTemporaryVertexPtr->root_PicElementPtr)->value;
}

void make_lowpass_image_uint8(uint8 *Y, REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, int32 NumPictureElements)
{
    int32 k;
    REGION_GRAPH_VERTEX *RegionGraphTemporaryVertexPtr;

    RegionGraphTemporaryVertexPtr = RegionGraphBaseVertexPtr;

    /* Loop over all NumPictureElements */
    for (k=0; k<NumPictureElements; k++)//, RegionGraphTemporaryVertexPtr++, Y++)
	{
		if (RegionGraphTemporaryVertexPtr[k].root_PicElementPtr < 0)
			Y[k]=RegionGraphTemporaryVertexPtr[k].value;
		else
			Y[k]=(RegionGraphBaseVertexPtr+RegionGraphTemporaryVertexPtr[k].root_PicElementPtr)->value;
	}
	/*	*Y = (RegionGraphTemporaryVertexPtr->root_PicElementPtr < 0) ?
             RegionGraphTemporaryVertexPtr->value : (RegionGraphBaseVertexPtr+RegionGraphTemporaryVertexPtr->root_PicElementPtr)->value;*/
}


/****************************************************************************/

/* GENERATE BANDPASS IMAGE */

void make_bandpass_image_int16(int16 *Y, uint8 *X1, uint8 *X2, int32 NumPictureElements)
{
    int32 k;

    for (k=0; k<NumPictureElements; k++)
        *(Y++) = (int16)*(X2++)-(int16)*(X1++);
}


/****************************************************************************/

/* DUPLICATE IMAGE */

void duplicate_image_uint8(uint8 *Y, uint8 *X, int32 NumPictureElements)
{
    int32 k;

    for (k=0; k<NumPictureElements; k++)
        *(Y++) = *(X++);
}


/****************************************************************************/

/* FUSE GRANULES IN BANDPASS IMAGE */

void make_fused_image_int16(int16 *Y, PICTURE_ELEMENT *PicElementPtr, int32 ConnectivityNdir, int32 imRows, int32 imCols,
                            bool edges)
{
    int32 m, n, k, off_m[8], off_n[8], curr_m, curr_n, new_m, new_n;
    int16 val, ext_val;
    pointer curr, last, pos, new_pos;
    PICTURE_ELEMENT *TempPicElementPtr;

    /* Set up offsets */
    off_m[0] = -1; off_m[1] = +1; off_m[2] =  0; off_m[3] =  0;
    off_m[4] = -1; off_m[5] = -1; off_m[6] = +1; off_m[7] = +1;
    off_n[0] =  0; off_n[1] =  0; off_n[2] = -1; off_n[3] = +1;
    off_n[4] = -1; off_n[5] = +1; off_n[6] = -1; off_n[7] = +1;

    /* Reset pointers */
    for (TempPicElementPtr=PicElementPtr, k=0; k<imRows*imCols; k++)
    {
        TempPicElementPtr->edge = FALSE;
        (TempPicElementPtr++)->m = -1;
    }

    /* Loop over NumPictureElements in image */
    for (pos=0, n=0; n<imCols; n++)
    {
        for (m=0; m<imRows; m++, pos++)
        {
            /* Test for a new region */
            if ((ext_val=Y[pos])!=0 && PicElementPtr[pos].m<0)
            {
                /* Set up pointers */
                curr = last = pos;
                curr_m = m;
                curr_n = n;

                /* Find all NumPictureElements of same sign in region */
                while (curr_m>=0)
                {
                    if (ext_val>0)
                    {
                        for (k=0; k<ConnectivityNdir; k++)
                        {
                            new_m = curr_m + off_m[k];
                            new_n = curr_n + off_n[k];

                            if (new_m>=0 && new_m<imRows && new_n>=0 && new_n<imCols)
                            {
                                new_pos = new_n*imRows+new_m;
                                val = Y[new_pos];
                                if (val>0)
                                {
                                    if (PicElementPtr[new_pos].m<0 && new_pos!=last)
                                    {
                                        PicElementPtr[last].m = new_m;
                                        PicElementPtr[last].n = new_n;
                                        last = new_pos;
                                        if (val>ext_val) ext_val = val;
                                    }
                                }
                                else
                                    PicElementPtr[curr].edge = TRUE;
                            }
                        }
                    }
                    else
                    {
                        for (k=0; k<ConnectivityNdir; k++)
                        {
                            new_m = curr_m + off_m[k];
                            new_n = curr_n + off_n[k];

                            if (new_m>=0 && new_m<imRows && new_n>=0 && new_n<imCols)
                            {
                                new_pos = new_n*imRows+new_m;
                                val = Y[new_pos];
                                if (val<0)
                                {
                                    if (PicElementPtr[new_pos].m<0 && new_pos!=last)
                                    {
                                        PicElementPtr[last].m = new_m;
                                        PicElementPtr[last].n = new_n;
                                        last = new_pos;
                                        if (val<ext_val) ext_val = val;
                                    }
                                }
                                else
                                    PicElementPtr[curr].edge = TRUE;
                            }
                        }
                    }

                    /* Move onto next PicElementPtr */
                    curr_m = PicElementPtr[curr].m;
                    curr_n = PicElementPtr[curr].n;
                    curr = curr_n*imRows+curr_m;
                }

                /* Set all NumPictureElements to the extreme value */
                if (!edges)
                {
                    curr_m = m; curr_n = n;
                    while (curr_m>=0)
                    {
                        new_pos = curr_n*imRows+curr_m;
                        Y[new_pos] = ext_val;
                        curr_m = PicElementPtr[new_pos].m;
                        curr_n = PicElementPtr[new_pos].n;
                    }
                }
                else
                {
                    curr_m = m; curr_n = n;
                    while (curr_m>=0)
                    {
                        new_pos = curr_n*imRows+curr_m;
                        if (PicElementPtr[new_pos].edge)
                            Y[new_pos] = ext_val;
                        else
                            Y[new_pos] = 0;
                        curr_m = PicElementPtr[new_pos].m;
                        curr_n = PicElementPtr[new_pos].n;
                    }
                }

                PicElementPtr[last].m = 0;

            }
        }
    }
}


/****************************************************************************/

/* ALLOCATE DYNAMIC MEMORY */

void allocate_memory(uint8 **imageIn, REGION_GRAPH_VERTEX **RegionGraphBaseVertexPtr, EXTREMAL_VERTEX **MinimaList, EXTREMAL_VERTEX **MaximaList,
                     uint8 **BandPassOutput1, uint8 **BandPassOutput2, PICTURE_ELEMENT **PicElementPtr, 
					 int32 **GranuleListBasePtr, int32 NumPictureElements, int32 out_type)
{
    int32 max_min_len;

    /* Allocate buffer for input image */
    *imageIn = (uint8 *)mxCalloc((unsigned int)NumPictureElements, sizeof(uint8));

    /* Allocate buffer for graph, one element per picture element*/
    *RegionGraphBaseVertexPtr = (REGION_GRAPH_VERTEX *)mxCalloc((unsigned int)NumPictureElements, sizeof(REGION_GRAPH_VERTEX));

    /* Allocate buffers for extrema lists, lists ordered by scale [0,1,2,..., NumPictureElements*/
    max_min_len = NumPictureElements; // /4+1; WATCH OUT
    *MaximaList = (EXTREMAL_VERTEX *)mxCalloc((unsigned int)max_min_len, sizeof(EXTREMAL_VERTEX));
    *MinimaList = (EXTREMAL_VERTEX *)mxCalloc((unsigned int)max_min_len, sizeof(EXTREMAL_VERTEX));
    (*MaximaList+max_min_len-1)->NextExtremum = -1; // used to prevent running off the end of allocated space
    (*MinimaList+max_min_len-1)->NextExtremum = -1; // used to prevent running off the end of allocated space

    /* Allocate buffers for bandpass outputs */
    if (out_type==BANDPASS || out_type==FUSEBPASS || out_type==EDGES)
    {
        *BandPassOutput1 = (uint8 *)mxCalloc((unsigned int)NumPictureElements, sizeof(uint8));
        *BandPassOutput2 = (uint8 *)mxCalloc((unsigned int)NumPictureElements, sizeof(uint8));

        if (out_type==FUSEBPASS || out_type==EDGES)
            *PicElementPtr = (PICTURE_ELEMENT *)mxCalloc((unsigned int)NumPictureElements, sizeof(PICTURE_ELEMENT));
        else
            *PicElementPtr = 0;
    }
    else
    {
        *BandPassOutput1 = 0;
        *BandPassOutput2 = 0;
    }

    /* Allocate buffer for granules */
    if (out_type==GRANULES)
	{
        *GranuleListBasePtr = (int32 *)mxCalloc((unsigned int)(GRANSIZE*10*NumPictureElements), sizeof(int32));
	}
    else
        *GranuleListBasePtr = 0;
}


/****************************************************************************/

/* CREATE MATLAB OUTPUT */

void create_matlab_output(mxArray **mY, int32 out_type, REGION_GRAPH_VERTEX *RegionGraphBaseVertexPtr, uint8 *BandPassOutput1,
                          uint8 *BandPassOutput2, int32 *GranuleListBasePtr, int32 *GranuleListCurrentPtr, int32 *GranuleCounter, 
						  PICTURE_ELEMENT *PicElementPtr, int32 imRows, int32 imCols, int32 ConnectivityNdir)
{
    switch (out_type)
    {
        case LOWPASS:
        {
            uint8 *Y;
            int dims[2];

            /* Allocate matrix for output image */
            dims[0] = imRows; dims[1] = imCols;
            *mY = mxCreateNumericArray(2, dims, mxUINT8_CLASS, mxREAL);
            Y = (uint8 *)mxGetPr(*mY);

            /* Copy low pass image to output matrix */
            make_lowpass_image_uint8(Y, RegionGraphBaseVertexPtr, imRows*imCols);

            break;
        }
        case BANDPASS: case FUSEBPASS: case EDGES:
        {
            int16 *Y;
            int dims[2];

            /* Allocate matrix for output image */
            dims[0] = imRows; dims[1] = imCols;
            *mY = mxCreateNumericArray(2, dims, mxINT16_CLASS, mxREAL);
            Y = (int16 *)mxGetPr(*mY);

            /* Get bandpass image */
            make_bandpass_image_int16(Y, BandPassOutput1, BandPassOutput2, imRows*imCols);
            if (out_type==FUSEBPASS || out_type==EDGES)
                make_fused_image_int16(Y, PicElementPtr, ConnectivityNdir, imRows, imCols, out_type==EDGES);

            break;
        }
        case GRANULES:
        {
            mxArray *mN, *mA, *mV, *mL, *mD, *mR, *mS, *mP, *mZ;
            double *N, *A, *V, *L, *D, *R, *S;
            int32 *Z, *GLtmp, num, index, area, value, k;
            int dims[2];
            char *fields[] = {"Number",
				              "area",
                              "value",
                              "level",
                              "deltaArea",
							  "last_area",
                              "root",
                              "PictureElement"};

            /* Calculate number of granules */
           /* num = 0;
            GLtmp = GranuleListBasePtr;
            while (GLtmp < GranuleListCurrentPtr)
            {
                num++;
                GLtmp += (*GLtmp) + 7;
            }*/
			num = *GranuleCounter;

            /* Allocate structure array for granules */
            *mY = mxCreateStructMatrix(1, 1, 8, (const char**)fields);

            /* Allocate each data array */
            mN = mxCreateDoubleMatrix(1, 1, mxREAL);
            N = mxGetPr(mN);
            mA = mxCreateDoubleMatrix(1, num, mxREAL);
            A = mxGetPr(mA);
            mV = mxCreateDoubleMatrix(1, num, mxREAL);
            V = mxGetPr(mV);
            mL = mxCreateDoubleMatrix(1, num, mxREAL);
            L = mxGetPr(mL);
            mD = mxCreateDoubleMatrix(1, num, mxREAL);
            D = mxGetPr(mD);           
			mR = mxCreateDoubleMatrix(1, num, mxREAL);
            R = mxGetPr(mR);
			mS = mxCreateDoubleMatrix(1, num, mxREAL);
            S = mxGetPr(mS);
            mP = mxCreateCellMatrix(1, num);

            /* Allocate each field in structure */
            mxSetFieldByNumber(*mY, 0, 0, mN);
            mxSetFieldByNumber(*mY, 0, 1, mA);
            mxSetFieldByNumber(*mY, 0, 2, mV);
            mxSetFieldByNumber(*mY, 0, 3, mL);
            mxSetFieldByNumber(*mY, 0, 4, mD);
            mxSetFieldByNumber(*mY, 0, 5, mR);
            mxSetFieldByNumber(*mY, 0, 6, mS);
            mxSetFieldByNumber(*mY, 0, 7, mP);

			// *N is a scalar setup by mxCreateDoubleMatrix above
			*N=*GranuleCounter;
            /* Store each granule */
            index = 0;
            GLtmp = GranuleListBasePtr;
            while (GLtmp < GranuleListCurrentPtr)
            {
                /* Set granule area */
                *(A++) = area = (double)*(GLtmp++);
				if (area==0)
					area=area;				
				if (area>((double)imRows*(double)imCols))
					area=area;

                /* Set granule value */
                *(V++) = (double)*(GLtmp++);

                /* Set extremum value */
                *(L++) = (double)*(GLtmp++);

                /* Set delta area value */
                *(D++) = (double)*(GLtmp++);

                /* Set root position */
                *(R++) = (double)*(GLtmp++);

                /* Set last start position */
                *(S++) = (double)*(GLtmp++);

                /* Set granule PictureElement */
                dims[0] = 1; dims[1] = area;
                mZ = mxCreateNumericArray(2, dims, mxINT32_CLASS, mxREAL);
                Z = (int32 *)mxGetPr(mZ);
                for (k=0; k<area; k++)
                    *(Z++) = *(GLtmp++) + 1; /* Allow for Matlab indexing */
                mxSetCell(mP, index, mZ);

                index++;
            }
            break;
		}
        case VERBOSE:
        {
            mxArray *RegionGraph, *mV, *mA, *mR, *mN, *mO, *mL;
            int32 j,i,num,index;
			double *V,*A;
			double *R,*N,*O;
			double *L;
            int dims[2];
            char *fields[] = {"value",
				              "AdjacencyFlags",
                              "root_PicElementPtr",
                              "next_PicElementPtr",
                              "OffsetToNextBoundaryPicElementPtr",
							  "last_area"};

			num = imRows*imCols;
			dims[0] = 1; dims[1] = 1;

            /* Allocate structure array for granules */
            *mY = mxCreateStructMatrix(1, 1, 6, (const char**)fields);
			            /* Allocate each data array */
		    mV = mxCreateDoubleMatrix(1,num, mxREAL);
		    V = mxGetPr(mV);
			mA = mxCreateDoubleMatrix(1,num, mxREAL);
			A = mxGetPr(mA);
	        mR = mxCreateDoubleMatrix(1,num, mxREAL);
			R = mxGetPr(mR);
	        mN = mxCreateDoubleMatrix(1,num, mxREAL);
	        N = mxGetPr(mN);
	        mO = mxCreateDoubleMatrix(1,num, mxREAL);
	        O = mxGetPr(mO);           
			mL = mxCreateDoubleMatrix(1,num, mxREAL);
	        L = mxGetPr(mL);
			index=0;
			mxSetFieldByNumber(*mY, 0, 0, mV);
            mxSetFieldByNumber(*mY, 0, 1, mA);
            mxSetFieldByNumber(*mY, 0, 2, mR);
            mxSetFieldByNumber(*mY, 0, 3, mN);
            mxSetFieldByNumber(*mY, 0, 4, mO);
            mxSetFieldByNumber(*mY, 0, 5, mL);

			for (j=0; j<num; j++)
			{
				V[j] = (double)RegionGraphBaseVertexPtr[j].value;
				A[j] = (double)RegionGraphBaseVertexPtr[j].AdjacencyFlags;
				if (false)
				{
					R[j] = (double)RegionGraphBaseVertexPtr[j].root_PicElementPtr; 
					N[j] = (double)RegionGraphBaseVertexPtr[j].next_PicElementPtr; 
					O[j] = (double)RegionGraphBaseVertexPtr[j].OffsetToNextBoundaryPicElementPtr;
				}
				else
				{
				if (RegionGraphBaseVertexPtr[j].root_PicElementPtr>=0)
					R[j] = (double)RegionGraphBaseVertexPtr[j].root_PicElementPtr+1; // positive values are pointers (offsets) convert to Matlab
				else
					R[j] = (double)RegionGraphBaseVertexPtr[j].root_PicElementPtr; // Minus values flag roots and the value is the areas.
				if (RegionGraphBaseVertexPtr[j].next_PicElementPtr>=0)
					N[j] = (double)RegionGraphBaseVertexPtr[j].next_PicElementPtr+1; // Matlab indexing
				else
					N[j] = (double)RegionGraphBaseVertexPtr[j].next_PicElementPtr-1; // Matlab indexing
				if (RegionGraphBaseVertexPtr[j].OffsetToNextBoundaryPicElementPtr>=0)
					O[j] = (double)RegionGraphBaseVertexPtr[j].OffsetToNextBoundaryPicElementPtr+1; // Matlab indexing?
				else
					O[j] = (double)RegionGraphBaseVertexPtr[j].OffsetToNextBoundaryPicElementPtr-1; // Matlab indexing?
				}
				L[j] = (double)RegionGraphBaseVertexPtr[j].last_area;
	            index++;
			}
			break;
		}
	}
}
